from sandbox import sdk2
from sandbox.common import errors as ce
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt
from sandbox.projects.common import file_utils as fu

from sandbox.projects.mail.Load.MobileApiAmmoGen import MobileApiAmmoGen
from sandbox.projects.mail.Load.ShootingComparison import ShootingComparison
from sandbox.projects.tank.load_resources.resources import YANDEX_TANKAPI_LXC_CONTAINER
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from sandbox.projects.tank.LoadTestResults import LoadTestResults

import yaml
import logging
import os.path
import requests
import time
import ast
import os

DefaultProfile = {
    "v1_messages": 14700,
    "v1_xlist": 13150,
    "v1_ava2": 8600,
    "v1_message_body": 8500,
    "v2_flags": 5600,
    "v1_attach": 3000,
    "v1_mark_read": 1500,
    "v1_settings": 1200,
    "v1_push": 1000,
    "v1_reset_fresh": 1000,
    "v2_reset_fresh": 650,
    "v1_uaz": 650,
    "v2_classification": 150,
    "v2_get_newsletters": 40,
    "v1_ava": 30,
    "v1_only_new": 10,
    "v1_set_parameters": 10
}


class MobileApiShooting(sdk2.Task):
    """ There will be a more meaningful comment """

    class Context(sdk2.Task.Context):
        ammo_url = ""
        ammo_generator_task_id = ""
        ammo_generator_find_id = ""
        shooting_task_id = ""
        shooting_find_id = ""
        comparison_task_id = ""
        send_comment_task_id = ""
        lxc_container = ""

    class Requirements(sdk2.Requirements):
        disk_space = 1024   # 1GiB on disk
        cores = 1           # exactly 1 core
        ram = 2048          # 2GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Shooting parameters') as tankapi_block:
            comment = sdk2.parameters.String('Comment on the shooting', default='Sandbox shooting')
            target = sdk2.parameters.String('Target ES', default='mail_mobapi_load.mobapi', required=True)
            ssl = sdk2.parameters.Bool('SSL shooting', default=True)
            scheduler = sdk2.parameters.List('Load profile', default=['{"duration": "300s", "type": "const", "ops": 200}'])
            pandora = sdk2.parameters.String('Pandora binary link', default='https://proxy.sandbox.yandex-team.ru/3176683213', required=True)
            profile = sdk2.parameters.JSON('Shooting profile', default=DefaultProfile, required=True)
            autostop = sdk2.parameters.List('Autostop requirements', default=['quantile(90,500ms,20s)', 'negative_http(2xx,20%,20s)'])
            component = sdk2.parameters.String('Regression component', default='')
            instances = sdk2.parameters.Integer('Pandora instances', default=2000, required=True)
            pgtype = sdk2.parameters.String('Type of the pg users', default='type500')

        with sdk2.parameters.Group('Uploaded files') as sources_block:
            load_file = sdk2.parameters.String('Template shooting config', default='https://proxy.sandbox.yandex-team.ru/3192338814')
            users_file = sdk2.parameters.String('Resource for ammo generator', default='https://proxy.sandbox.yandex-team.ru/2533570206')

        with sdk2.parameters.Group('The choice of the tank') as tank_coice_block:
            with sdk2.parameters.RadioGroup('Use tank', required=True) as use_tank:
                use_tank.values['mimas.tanks.mail.yandex.net'] = use_tank.Value(value='MIMAS')
                use_tank.values['rhea.tanks.mail.yandex.net'] = use_tank.Value(value='RHEA', default=True)
                use_tank.values['iapet.tanks.mail.yandex.net'] = use_tank.Value(value='IAPET')

        with sdk2.parameters.Group('Parameters for comparing of the shooting') as comparing_block:
            comparing = sdk2.parameters.Bool('To comparing the shooting?', default=True)
            with comparing.value[True]:
                rfile = sdk2.parameters.String('File with the reference shootings', default='')
                threshold = sdk2.parameters.Integer('Permissible deviation in %', default=5)
                ticket = sdk2.parameters.String('Ticket for comment', default='MMAPI-155')
                st_token = sdk2.parameters.String('Vault record name with startrek token', default='robot-mail-internal-startrek-token')

        with sdk2.parameters.Output:
            lunapark_job_id = sdk2.parameters.String('Lunapark job id', default_value='')
            lunapark_link = sdk2.parameters.String('Lunapark link', default_value='')

    def start_shooting(self, desc, config_content):
        tanks = []
        tanks.append(self.Parameters.use_tank)
        container = YANDEX_TANKAPI_LXC_CONTAINER.find(state=ctr.State.READY).order(-YANDEX_TANKAPI_LXC_CONTAINER.id).first().id
        if container is not None:
            self.Context.lxc_container = str(container)
        shooting_task = ShootViaTankapi(
            self,
            description=desc,
            ammo_source='in_config',
            config_source='file',
            config_content=config_content,
            tanks=tanks,
            container=container
        ).enqueue()
        self.loger.info('Subtask with shooting is started')
        self.Context.shooting_task_id = str(shooting_task.id)
        raise sdk2.WaitTask([shooting_task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=14400)

    def find_shooting(self):
        shooting_find = sdk2.Task.find(
            id=self.Context.shooting_task_id,
            children=True
        ).first()
        self.Context.shooting_find_id = str(shooting_find.id)
        self.loger.info('found ammo_generator_task_id = %s', shooting_find.id)
        self.Parameters.lunapark_job_id = shooting_find.Parameters.lunapark_job_id
        self.Parameters.lunapark_link = shooting_find.Parameters.lunapark_link

    def make_ammo(self):
        self.loger.info('Start ammo generation')
        ammo_gen_task = MobileApiAmmoGen(
            self,
            description='ammo generator for task {0}'.format(self.id),
            source=self.Parameters.users_file,
            profile=self.Parameters.profile,
            pgtype=self.Parameters.pgtype,
            ).enqueue()

        self.Context.ammo_generator_task_id = str(ammo_gen_task.id)
        self.Context.save()
        raise sdk2.WaitOutput({ammo_gen_task.id: 'ammo'}, wait_all=True, timeout=3600)

    def get_ammo(self):
        ammo_gen_find = self.find(id=self.Context.ammo_generator_task_id).first()
        self.Context.ammo_generator_find_id = str(ammo_gen_find.id)
        self.loger.info('found ammo_generator_task_id = %s', ammo_gen_find.id)
        return ammo_gen_find.Parameters.ammo

    def make_conf(self):
        autostop = []
        scheduler = []
        config_content = fu.read_file(get_source(self.Parameters.load_file, 'load.yaml'))
        yaml_config = yaml.safe_load(config_content)
        # Modify config for current task
        yaml_config['pandora']['pandora_cmd'] = self.Parameters.pandora.encode('utf-8')
        yaml_config['pandora']['config_content']['pools'][0]['gun']['target'] = ':'.join([self.es_to_fqdn(self.Parameters.target).encode('utf-8'), '443' if self.Parameters.ssl else '80'])
        yaml_config['pandora']['config_content']['pools'][0]['gun']['type'] = 'mobileapi'
        yaml_config['pandora']['config_content']['pools'][0]['gun']['ssl'] = self.Parameters.ssl
        yaml_config['pandora']['config_content']['pools'][0]['startup']['times'] = self.Parameters.instances
        yaml_config['pandora']['config_content']['pools'][0]['ammo']['source']['path'] = "./ammo.json"
        yaml_config['pandora']['config_content']['pools'][0]['ammo']['source']['type'] = "file"
        yaml_config['pandora']['config_content']['pools'][0]['ammo']['type'] = "mobileapi_provider"
        yaml_config['pandora']['resources'][0]['src'] = self.Context.ammo_url
        yaml_config['pandora']['resources'][0]['dst'] = './ammo.json'
        yaml_config['yasm']['panels']['mobile-api']['tags'] = "itype=mail;stage=mail_mobapi_load;deploy_unit={}".format(self.Parameters.target.split('.')[1]).encode('utf-8')
        yaml_config['yasm']['panels']['tank']['tags'] = "itype=mail_load;stage=mail_load_tanks;deploy_unit={}".format(self.Parameters.use_tank.split('.')[0]).encode('utf-8')
        yaml_config['uploader']['job_name'] = "[mobile-api][{}][{}]".format(self.Parameters.scheduler, self.Parameters.comment).encode('utf-8')

        if self.Parameters.component:
            yaml_config['uploader']['component'] = self.Parameters.component.encode('utf-8')

        for condition in self.Parameters.autostop:
            autostop.append(condition.encode('utf-8'))
        yaml_config['autostop']['autostop'] = autostop

        for rps in self.Parameters.scheduler:
            scheduler.append(ast.literal_eval(rps.encode('utf-8')))
        yaml_config['pandora']['config_content']['pools'][0]['rps'] = scheduler
        self.loger.info("scheduler is {}".format(yaml_config['pandora']['config_content']['pools'][0]['rps']))

        return yaml.dump(yaml_config, default_flow_style=False, encoding=None)

    def comparing_shooting(self):
        comparison_task = ShootingComparison(
            self,
            sid=self.Parameters.lunapark_job_id,
            rfile=self.Parameters.rfile
        ).enqueue()

        self.Context.comparison_task_id = str(comparison_task.id)
        raise sdk2.WaitTask([comparison_task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=14400)

    def comparison_result(self):
        # Search task with comparison of self shooting
        comparison_find = sdk2.Task.find(
            id=self.Context.comparison_task_id,
            children=True
        ).first()
        # Get results of the comparison
        comparison_result = comparison_find.Parameters.comparison_result
        shooting_type = comparison_find.Parameters.shooting_type
        # Build comment for ticket
        header = "Regression"
        footer = "https://wiki.yandex-team.ru/Load/"
        comment_for_ticket = "((@lix0 Load Support))"
        comparison_status = comparison_result.split("\n")[-1]
        comparison_body = "\n".join(comparison_result.split("\n")[:-1])
        comment_for_ticket += "\n<{%s\n#|\n||%s||\n|#\n%s}>\n%s" % (
            header,
            comparison_body.replace("\n", "||\n||").replace("passed", "!!(green)passed!!").replace("failed", "!!(red)failed!!").replace("improved", "!!(yellow)improved!!"),
            footer,
            comparison_status.replace("passed", "**!!(green)SUCCESS!!**").replace("failed", "**!!(red)FAILED!!**")
            )
        self.Context.comment_for_ticket = comment_for_ticket
        # Send comment into ticket
        send_comment_task = LoadTestResults(
            self,
            shoot_id=self.Parameters.lunapark_job_id,
            report_type=shooting_type,
            additional_message=comment_for_ticket,
            ticket_id=self.Parameters.ticket,
            st_token_name=self.Parameters.st_token,
            send_comment=True
        ).enqueue()

        self.Context.send_comment_task_id = str(send_comment_task.id)
        self.loger.info('Subtask with shooting is started')
        raise sdk2.WaitTask([send_comment_task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=14400)

    def on_execute(self):
        self.loger = logger()
        self.loger.info("Start Mobile-Api shooting")

        desc = self.Parameters.comment

        with self.memoize_stage.make_ammo:
            self.make_ammo()

        with self.memoize_stage.make_conf:
            ammo_resource = self.get_ammo()
            self.Context.ammo_url = str(ammo_resource.http_proxy)
            config_content = self.make_conf()

        with self.memoize_stage.shooting:
            self.start_shooting(desc, config_content)

        if self.Context.shooting_task_id != "":
            self.find_shooting()
        # Evaluate shooting and send comment with the result of evaluation into ticket
        if self.Parameters.comparing and self.Parameters.lunapark_job_id != "":
            if self.Context.comparison_task_id == "":
                self.comparing_shooting()
            elif self.Context.send_comment_task_id == "":
                self.comparison_result()
            # ==== End Of Class ==== #

    def es_to_fqdn(self, endpoint_set_id):
        dcs = ['iva', 'myt', 'sas', 'vla']
        fqdns = []

        for dc in dcs:
            res = requests.get('http://sd.yandex.net:8080/resolve_endpoints/json', json={
                'cluster_name': dc,
                'endpoint_set_id': endpoint_set_id,
                'client_name': 'mail_load',
            })
            print(res.json())
            if 'endpoint_set' in res.json():
                if 'endpoints' in res.json()['endpoint_set']:
                    for endpoint in res.json()['endpoint_set']['endpoints']:
                        fqdns.append(endpoint['fqdn'])

        if len(fqdns) == 0:
            raise ce.TaskFailure(
                "[ERROR] Cannot get endpoints for endpoint_set {es}".format(es=endpoint_set_id))

        return fqdns[0]


def logger():
    loggerr = logging.getLogger('%s_%s' % (__name__, time.time()))
    loggerr.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s %(levelname)s [%(processName)s: %(threadName)s] %(message)s')
    file_handler = logging.handlers.RotatingFileHandler(
        'mobileapi_shooting.log',
        maxBytes=(1024 * 1024),
        backupCount=5
    )

    file_handler.setLevel(logging.DEBUG)
    file_handler.setFormatter(formatter)
    loggerr.addHandler(file_handler)
    return loggerr


def get_source(url, dst):
    session = requests.session()
    try:
        with open(dst, 'wb') as resource:
            resource.write(session.get(url, stream=True).content)
        return os.path.abspath(dst)
    except Exception as ex:
        raise ce.TaskFailure("Can't download resource. {}".format(ex))
    finally:
        session.close()
