from sandbox import sdk2
from sandbox.common import errors as ce
from sandbox.common.types import resource as ctr

import yaml
import logging
import json
import os.path
import requests
import time
import ast
import os
import re

class CompError(Exception):
    def __init__(self, text):
        self.txt = text

class ShootingComparison(sdk2.Task):
    """ There will be a more meaningful comment """

    class Context(sdk2.Task.Context):
        comparison_file = ""
        scheduler = ""
        service_name = ""
        imbalance_rps = 0

    class Requirements(sdk2.Requirements):
        disk_space = 1024   # 1GiB on disk
        cores = 1           # exactly 1 core
        ram = 1024          # 1GiB or less

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Parameters for comparison') as comparison_block:
            sid = sdk2.parameters.String('Shooting ID', default="", required=True)
            rfile = sdk2.parameters.String('File with the list of shootings', default="")
            fyaml = sdk2.parameters.Bool('File is yaml format?', default=False)
            rid = sdk2.parameters.String('Regression ID', default="")
            rname = sdk2.parameters.String('Regression name', default="")
            threshold = sdk2.parameters.Integer('Permissible deviation in %', default=5, required=True)
            wmi = sdk2.parameters.String('Shootings aggregator', default='https://lunapark.yandex-team.ru', required=True)

        # with sdk2.parameters.Group('Parameters for comment') as comment_block:
        #     send_comment = sdk2.parameters.Bool('Send comment into ticket?', default=True)
        #     with send_comment.value[True]:
        #         ticket = sdk2.parameters.String('Ticket for comment', default='MAILPG-2716')
        #         tracker = sdk2.parameters.String('Tickets tracker', default='https://st-api.yandex-team.ru', required=True)

        with sdk2.parameters.Group('What tests will be execute?') as tests_block:
            CPU = sdk2.parameters.Bool('Check for CPU usage?', default=True)
            NETMB = sdk2.parameters.Bool('Check for network traffic?', default=True)
            Q50 = sdk2.parameters.Bool('Check the timings at the 50-th quantile?', default=True)
            Q75 = sdk2.parameters.Bool('Check the timings at the 75-th quantile?', default=True)
            Q80 = sdk2.parameters.Bool('Check the timings at the 80-th quantile?', default=True)
            Q85 = sdk2.parameters.Bool('Check the timings at the 85-th quantile?', default=True)
            Q90 = sdk2.parameters.Bool('Check the timings at the 90-th quantile?', default=True)
            Q95 = sdk2.parameters.Bool('Check the timings at the 95-th quantile?', default=False)
            Q98 = sdk2.parameters.Bool('Check the timings at the 98-th quantile?', default=False)

        with sdk2.parameters.Output:
            comparison_result = sdk2.parameters.String('Comparison result', default_value="")
            shooting_type = sdk2.parameters.String('Type of shooting compared', default_value="")

# Data of the shooting
    def job_summary(self):
        try:
            self.shooconf, self.shootype, self.sname = get_scheduler(self.Parameters.wmi, self.Parameters.sid)
            self.Context.scheduler = self.shooconf
            self.Parameters.shooting_type = self.shootype
            self.Context.service_name = self.sname
            self.imbalance_rps = float(api_job(self.Parameters.wmi, self.Parameters.sid, 'summary')[0]['imbalance_rps'])
            self.Context.imbalance_rps = self.imbalance_rps
        except Exception as ex:
            raise CompError("Impossible to get shooting parameters.\nError:{}".format(ex))

    def job_percentiles(self):
        jobpercentiles = api_job(self.Parameters.wmi, self.Parameters.sid, 'percentiles')
        self.percentiles = parse_percentiles(jobpercentiles)

    def job_monitoring(self):
        if self.sname == None:
            self.monitoring = None
        else:
            jobmonitoring = api_job(self.Parameters.wmi, self.Parameters.sid, 'monitoring')
            self.monitoring = parse_monitoring(jobmonitoring, self.sname)

    def job_http(self):
        self.http = {}
        jobhttp = api_job(self.Parameters.wmi, self.Parameters.sid, 'http')

        if type(jobhttp) == list and len(jobhttp) > 0:

            for item in jobhttp:
                http_code = item[u'http']
                self.http[http_code] = item[u'percent']
        else:
            self.test[u'http_answers'] = False
            self.reason[u'http_answers'] = "Has no http answers data!"

    def job_net(self):
        self.net = {}
        jobnet = api_job(self.Parameters.wmi, self.Parameters.sid, 'net')

        if type(jobnet) == list and len(jobnet) > 0:

            for item in jobnet:
                net_code = item[u'net']
                self.net[net_code] = item[u'percent']
        else:
            self.test[u'net_answers'] = False
            self.reason[u'net_answers'] = "Has no net answers data!"

# Generate shooting's list for comparing
# Get list from regression component
    def job_list(self):
        complist = []
        templist = api_regress(self.Parameters.wmi, self.Parameters.rid, 'joblist')

        if type(templist) == list and len(templist) > 0:
            for item in templist:
                number = item[u'n']
                if get_scheduler(self.Parameters.wmi, number)[0] == self.shooconf:
                    complist.append(number)
                else:
                    continue

        return sorted(complist)[-10:]

# Get list from file
    def ref_list(self):
        if self.Parameters.fyaml:
            complist = self.list_from_yaml()
        else:
            complist = self.list_from_file()
        return complist

# For YAML files
    def list_from_yaml(self):
        self.Context.complist = "Shootings from file: "
        complist = []
        rfile = get_source(self.Parameters.rfile, 'comparison.yaml')
        self.Context.comparison_file = rfile
        with open(rfile) as rfile:
            try:
                reference = yaml.safe_load(rfile)
                for shooting in reference[self.shootype]:
                    # shooting = shoot.replace("\n","").replace(" ","")
                    # self.Context.complist += "," + shooting
                    if get_scheduler(self.Parameters.wmi, shooting)[0] == self.shooconf:
                        complist.append(shooting)
                    else:
                        continue
                return complist
            except Exception as ex:
                raise CompError("Error during reading yaml file.\n{}".format(ex))

# For other files
    def list_from_file(self):
        self.Context.complist = "Shootings from file: "
        complist = []
        rfile = get_source(self.Parameters.rfile, 'comparison.list')
        self.Context.comparison_file = rfile
        with open(rfile) as rfile:
            for line in rfile:
                shooting = line.replace("\n","").replace(" ","")
                self.Context.complist += "," + shooting
                if get_scheduler(self.Parameters.wmi, shooting)[0] == self.shooconf:
                    complist.append(shooting)
                else:
                    continue
        return complist

# Reference data for comparing
    def data_for_comparing(self):
        self.ref_percentiles = {}
        self.ref_monitoring = {}

        for shoot in self.complist:
            self.ref_percentiles[shoot] = parse_percentiles(api_job(self.Parameters.wmi, int(shoot), 'percentiles'))

            if self.sname == None:
                self.ref_monitoring[shoot] = None
            else:
                self.ref_monitoring[shoot] = parse_monitoring(api_job(self.Parameters.wmi, int(shoot), 'monitoring'), self.sname)

# Add shooting to regression
    def add_to_regression(self):
        if self.Parameters.rname == "":
            pass
        else:
            try:
                _ = add_job(self.Parameters.wmi, self.Parameters.sid, self.Parameters.rname)
                self.Context.add_to_regression = "Shooting {} was add to regression {}".format(self.Parameters.sid, self.Parameters.rname)
            except Exception as ex:
                self.Context.add_to_regression = "Error {}".format(ex)

# Link to luna comparing sheet
    def luna_compare(self, num):
        lunatemplate = 'https://lunapark.yandex-team.ru/compare/#jobs=%s&tab=test_data&mainjob=%s&helper=all&cases=&plotGroup=additional&metricGroup=&target='
        compared = sorted(self.complist)[-num:]
        compared.insert(0, self.Parameters.sid)
        return lunatemplate%(",".join(str(i).replace("\n","").replace(" ","") for i in compared), self.Parameters.sid)

# Tests
# 0 - test failed ; 1 - test passed ; 2 - improved perfomance

    def check_imbalance(self):
        imbalance_rps = []

        for item in self.complist:
            imbalance_rps.append(api_job(self.Parameters.wmi, item, 'summary')[0]['imbalance_rps'])
        mediana = get_median(imbalance_rps)

        if mediana == None:
            self.test[u'check_imbalance'] = 0
            self.reason[u'check_imbalance'] = 'Regression list is empty'
        elif self.imbalance_rps == 0:
            self.test[u'check_imbalance'] = 1
        else:
            deviation = mediana / self.imbalance_rps

            if deviation > self.threshold:
                self.test[u'check_imbalance'] = 0
                self.reason[u'check_imbalance'] = 'Shooting is unbalanced faster in %f times'%(deviation)
            elif (float(1)/deviation) > self.threshold:
                self.test[u'check_imbalance'] = 2
                self.reason[u'check_imbalance'] = 'Shooting is unbalanced slowly in %f times'%(float(1)/deviation)
            else:
                self.test[u'check_imbalance'] = 1

    def test_http(self):
        self.job_http()
        if u'http_answers' not in self.test.keys():
            try:
                deviation = float(100) - self.http[200]

                if deviation == float(0):
                    self.test[u'http_answers'] = 1
                else:
                    self.test[u'http_answers'] = 0
                    self.reason[u'http_answers'] = '%f percents wrong http answers'%(deviation)

            except Exception:
                self.test[u'http_answers'] = 0
                self.reason[u'http_answers'] = 'Has no successfull http answers'

    def test_net(self):
        self.job_net()
        if u'net_answers' not in self.test.keys():
            try:
                deviation = float(100) - self.net[0]

                if deviation == float(0):
                    self.test[u'net_answers'] = 1
                else:
                    self.test[u'net_answers'] = 0
                    self.reason[u'net_answers'] = '%f percents wrong network answers'%(deviation)

            except Exception:
                self.test[u'http_answers'] = 0
                self.reason[u'http_answers'] = 'Has no successfull net answers'

    def test_quantile(self, quantile, band):
        arrayq = []

        for item in self.complist:
            arrayq.append(self.ref_percentiles[item][quantile])

        mediana = get_median(arrayq)
        if mediana == None:
            self.test[u'q' + quantile] = 0
            self.reason[u'q' + quantile] = 'Regression list is empty'

        elif self.percentiles[quantile] == 0:
            self.test[u'q' + quantile] = 0
            self.reason[u'q' + quantile] = 'Shooting\'s data is absent'

        else:
            deviation = self.percentiles[quantile] / mediana

            if deviation > float(band):
                self.test[u'q' + quantile] = 0
                self.reason[u'q' + quantile] = 'Longer by %f percents'%((deviation - float(1))*100)

            elif (float(1)/deviation) > float(band):
                self.test[u'q' + quantile] = 2
                self.reason[u'q' + quantile] = 'Faster by %f percents'%(((float(1)/deviation) - float(1))*100)

            else:
                self.test[u'q' + quantile] = 1

    def test_monitoring(self, metric, measure, band):

        if self.sname == None:
            self.test[metric] = 1
            self.reason[metric] = "Monitoring is not configured"
        else:

            try:
                indexes = {u'avg':0, u'min':1, u'max':2}
                index = indexes[measure]
                signals = {
                    u'cpu_usage':u'custom:portoinst-cpu_usage_cores_tmmv',
                    u'cpu_wait':u'custom:portoinst-cpu_wait_cores_tmmv',
                    u'io_read':u'custom:portoinst-io_read_fs_bytes_tmmv',
                    u'io_write':u'custom:portoinst-io_write_fs_bytes_tmmv',
                    u'net_mb_summ':u'custom:portoinst-net_mb_summ',
                    u'memory_usage':u'custom:portoinst-memory_usage_gb_tmmv'
                }
                signal = signals[metric]
                arraym = []

                for item in self.complist:
                    arraym.append(self.ref_monitoring[item][signal][index])
                mediana = get_median(arraym)

                if mediana == None:
                    self.test[metric] = 0
                    self.reason[metric] = 'Regression list is empty'
                elif self.monitoring[signal][index] == 0:
                    self.test[metric] = 0
                    self.reason[metric] = 'Shooting\'s data is absent'
                else:
                    deviation = self.monitoring[signal][index] / mediana

                    if deviation > float(band):
                        self.test[metric] = 0
                        self.reason[metric] = 'Worse by %f percents'%((deviation - float(1))*100)
                    elif (float(1)/deviation) > float(band):
                        self.test[metric] = 2
                        self.reason[metric] = 'Better by %f percents'%(((float(1)/deviation) - float(1))*100)
                    else:
                        self.test[metric] = 1

            except Exception as ex:
                self.test[metric] = 0
                self.reason[metric] = "Monitoring test was received error: {}".format(ex)

    def test_Q50(self):
        if self.Parameters.Q50:
            self.test_quantile(u'50', self.threshold)

    def test_Q75(self):
        if self.Parameters.Q75:
            self.test_quantile(u'75', self.threshold)

    def test_Q80(self):
        if self.Parameters.Q80:
            self.test_quantile(u'80', self.threshold)

    def test_Q85(self):
        if self.Parameters.Q85:
            self.test_quantile(u'85', self.threshold)

    def test_Q90(self):
        if self.Parameters.Q90:
            self.test_quantile(u'90', self.threshold)

    def test_Q95(self):
        if self.Parameters.Q95:
            self.test_quantile(u'95', self.threshold)

    def test_Q98(self):
        if self.Parameters.Q98:
            self.test_quantile(u'98', self.threshold)

    def test_CPU(self):
        if self.Parameters.CPU:
            self.test_monitoring(u'cpu_usage', u'max', self.threshold)

    def test_netmb(self):
        if self.Parameters.NETMB:
            self.test_monitoring(u'net_mb_summ', u'avg', self.threshold)

# Test analyze
    def get_result(self):
        self.passed = True
        comparison_result = "TEST | RESULT"

        for key in sorted(self.test.keys()):
            if self.test[key] == 0:
                self.passed = False
                comparison_result += "\n{} | {} | {}".format(key, "failed", self.reason[key])
            elif self.test[key] == 2:
                comparison_result += "\n{} | {} | {}".format(key, "improved", self.reason[key])
            else:
                comparison_result += "\n{} | {}".format(key, "passed")

# Add links to shootings comparison in lunapark
        comparison_result += "\nLinks to comparison in lunapark"
        for count in (2,5):
            comparison_result += "\nCompare {} | {}".format(count, self.luna_compare(count))

        if self.passed == True:
            self.add_to_regression()
            comparison_result += "\npassed"
        else:
            comparison_result += "\nfailed"

        self.Parameters.comparison_result = comparison_result

# Run the task
    def on_execute(self):
        self.loger = logger()
        self.loger.info("Start comparison")

# Check the presence of the shooting number and get a sign of compliance to form a comparison list
        if self.Parameters.sid == "":
            raise ce.TaskFailure("Shooting is not set")
        else:

# Ok, lets try to test something
            try:
                self.job_summary()
                self.complist = []
                self.compare_sources = 0

# Check availability of conditions for formation of the list of comparison. If they are available we get a list for comparison
                if self.Parameters.rfile == "" and self.Parameters.rid == "":
                    raise CompError("No comparison data specified")
                if self.Parameters.rid != "":
                    self.complist += self.job_list()
                    self.compare_sources += 1
                if self.Parameters.rfile != "":
                    self.complist += self.ref_list()
                    self.compare_sources += 2
                if len(self.complist) == 0:
                    raise CompError("Comparing list is empty")

# Initializing tracking dictionaries and data
                self.test = {}
                self.reason = {}
                self.threshold = float(1) + float(self.Parameters.threshold)/float(100)
                self.job_percentiles()
                self.job_monitoring()

# Constructor of the test cases
# For line shootings check only the value of RPS imbalance
                if self.shootype == "line":
                    self.check_imbalance()

# For const shootings check the numerical parameters
                else:
                    self.data_for_comparing()
                    self.test_http()
                    self.test_net()
                    self.test_Q50()
                    self.test_Q75()
                    self.test_Q80()
                    self.test_Q85()
                    self.test_Q90()
                    self.test_Q95()
                    self.test_Q98()
                    self.test_CPU()
                    self.test_netmb()

# Print the result of the comparing
                self.get_result()

# If something goes wrong
            except CompError as comperror:
                self.Parameters.comparison_result = "Regression test failed.\n{}\nfailed".format(comperror.txt)

############# End Of Class #############

def logger():
    loggerr = logging.getLogger('%s_%s' % (__name__, time.time()))
    loggerr.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s %(levelname)s [%(processName)s: %(threadName)s] %(message)s')
    file_handler = logging.handlers.RotatingFileHandler(
        'shooting_comparison.log',
        maxBytes=(1024 * 1024),
        backupCount=5
    )

    file_handler.setLevel(logging.DEBUG)
    file_handler.setFormatter(formatter)
    loggerr.addHandler(file_handler)
    return loggerr

# Download sandbox resource
def get_source(url, dst):
    session = requests.session()
    try:
        with open(dst, 'wb') as resource:
            resource.write(session.get(url, stream=True).content)
        return os.path.abspath(dst)
    except Exception as ex:
        raise ce.TaskFailure("Can't download resource. {}".format(ex))
    finally:
        session.close()

# Generate order and usability dictionary from response of the API request
# For perccentiles data
def parse_percentiles(perclist):
    percentiles = {}

    if type(perclist) == type([]) and len(perclist) > 7:
        for item in perclist[:7]:
            quantile = item[u'percentile']
            percentiles[quantile] = float(item[u'ms'])
    else:
        raise CompError('Wrong percentiles!')

    return percentiles

# For monitoring data
def parse_monitoring(monlist, service):
    monitoring = {}

    if type(monlist) == type([]) and len(monlist) > 0 :
        for item in monlist:
            if item[u'host'] == service:
                metric = item[u'metric']
                monitoring[metric] = (item[u'avg'], item[u'min'], item[u'max'])
            else:
                continue
    else:
        pass

    return monitoring

# Request to Lunapark for job data
def api_job (host, id, method):

    methods = {
        'summary':'summary.json',
        'times':'dist/times.json',
        'percentiles':'dist/percentiles.json',
        'http':'dist/http.json',
        'net':'dist/net.json',
        'cases':'dist/cases.json',
        'monitoring':'monitoring.json'
    }

    try:
        response = requests.get('%s/api/job/%s/%s'%(host, id, methods[method]))
        return response.json()
    except Exception as ex:
        raise CompError('Wrong API job request. %s'%(ex))
    finally:
        requests.session().close()

# Get values for scheduler shooting's type and name of service from shooting's config
def get_scheduler(host, sid):
# Set default values
    scheduler = None
    shootype = None
    servicename = None

# Generate and execute API request to Lunapark
    URL = str(host) + "/api/job/" + str(sid) + "/configinfo.txt"
    session = requests.session()
    response = session.get(URL)
    if response.status_code == 200:
        config = yaml.safe_load(response.content)

# Check YASM section in shooting's config
        try:
            panels = config['yasm']['panels'].keys()
            yasm = True
        except Exception:
            yasm = False

# If shooting was via phantom
        try:
            if config['phantom']['enabled'] == True:

                scheduler = config['phantom']['load_profile']['schedule'].replace(" ","")
                shootype = scheduler.split('(')[0]

                if yasm == True:
                    for panel in panels:
                        if re.compile(panel).search(config['phantom']['address']) != None:
                            servicename = panel

# If shooting was via pandora
            elif config['pandora']['enabled'] == True:

                if type(config['pandora']['config_content']['pools'][0]['rps']) == list:
                    scheduler = config['pandora']['config_content']['pools'][0]['rps'][0]
                else:
                    scheduler = config['pandora']['config_content']['pools'][0]['rps']
                shootype = scheduler['type']
                scheduler = ','.join([':'.join((key, str(scheduler[key]))) for key in sorted(scheduler, reverse=True)])

# If YASM monitoring is use then get a service name for monitoring tests
                if yasm == True:
                    for panel in panels:
                        if re.compile(panel).search(config['pandora']['config_content']['pools'][0]['gun']['target']) != None:
                            servicename = panel
            else:
                pass
        except Exception:
            pass
    else:
        pass

    session.close()
    return (scheduler, shootype, servicename)


# Add shooting to regression. May be will update to all cases of modifying of a shooting
def add_job(host, id, rname):
    payload = {'regression':1, 'component':rname}

    try:
        response = requests.post('%s/api/job/%s/edit.json'%(host, id), data=json.dumps(payload))
    except Exception as ex:
        raise CompError('Job is not add to regression. %s'%(ex))
    finally:
        requests.session().close()


# Request to Lunapark for regression data
def api_regress (host, id, method):
    methods = {
        'components':'componentlist.json',
        'kpilist':'kpilist.json',
        'joblist':'joblist.json'
    }

    try:
        response = requests.get('%s/api/regress/%s/%s'%(host, id, methods[method]))
        return response.json()
    except Exception, ex:
        raise CompError('Wrong API regression request. %s'%(ex))
    finally:
        requests.session().close()


# Calculate of the median value
def get_median(array):
    lenght = len(array)

    if int(lenght) > 5:
        return sum(sorted(array)[1:-1])/float(lenght - 2)
    elif int(lenght) > 0:
        return sum(sorted(array))/float(lenght)
    else:
        return None
