from sandbox import sdk2
from sandbox.common import errors as ce
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt
from sandbox.projects.common import file_utils as fu
from sandbox.projects.tank.load_resources.resources import AMMO_FILE
from sandbox.projects.tank.load_resources.resources import YANDEX_TANKAPI_LXC_CONTAINER
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from sandbox.projects.tank.LoadTestResults import LoadTestResults

import yaml
import logging
import json
import os.path
import random
import requests
import subprocess
import time
import ast
import os

defaultProfile = {"store_1000": 1}
legalHandlers = ["store", "append", "checkspam", "collect", "save", "sendmail", "sendsystemmail"]


class SmtpgateShooting(sdk2.Task):

    class Context(sdk2.Task.Context):
        ammo_url = ""
        ammo_resource = ""
        clear_command = ""
        clear_result = False
        lxc_container = ""
        send_comment_task_id = ""
        shooting_task_id = ""
        shooting_find_id = ""
        shootype = ""

    class Requirements(sdk2.Requirements):
        disk_space = 1024   # 1GiB on disk
        cores = 1           # exactly 1 core
        ram = 2048          # 2GiB or less

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Shooting parameters') as tankapi_block:
            ticket = sdk2.parameters.String('Lead ticket', required=True)
            comment = sdk2.parameters.String('Comment on the shooting', default='Sandbox shooting')
            host = sdk2.parameters.String('Target host', default='smtp-loadtest2p.mail.yandex.net:2000', required=True)
            scheduler = sdk2.parameters.List('Load profile', default=['{"duration": "300s", "type": "const", "ops": 100}'], required=True)
            profile = sdk2.parameters.JSON('Shooting profile', default=defaultProfile, required=True)
            autostop = sdk2.parameters.List('Autostop requirements', default=['quantile(90,500ms,20s)', 'negative_http(2xx,20%,20s)'])
            instances = sdk2.parameters.Integer('Pandora instances', default=2000, required=True)
            st_token = sdk2.parameters.String('Vault record name with startrek token', default='lunapark-startrek-token')

        with sdk2.parameters.Group('Uploaded files') as sources_block:
            pandora = sdk2.parameters.String('Pandora binary link', default='https://proxy.sandbox.yandex-team.ru/1450932457', required=True)
            load_file = sdk2.parameters.String('Template shooting config', default='https://proxy.sandbox.yandex-team.ru/1406976897', required=True)
            users_file = sdk2.parameters.String('Resource for ammo generator', default='https://proxy.sandbox.yandex-team.ru/1406989146', required=True)
            monitoring_file = sdk2.parameters.String('File for telegraf monitoring', default='https://proxy.sandbox.yandex-team.ru/1407004432', required=True)

        with sdk2.parameters.Group('The choice of the tank') as tank_coice_block:
            with sdk2.parameters.RadioGroup('Use tank', required=True) as use_tank:
                use_tank.values['mimas.tanks.mail.yandex.net'] = use_tank.Value(value='MIMAS', default=True)
                use_tank.values['rhea.tanks.mail.yandex.net'] = use_tank.Value(value='RHEA')
                use_tank.values['iapet.tanks.mail.yandex.net'] = use_tank.Value(value='IAPET')

        with sdk2.parameters.Group('Default values for ammo') as defammo_block:
            defsize = sdk2.parameters.String('Default size of letter', description="Defines the size of the message if no other is specified", default='1000', required=True)
            defcount = sdk2.parameters.Integer('Count of bullits in ammo file', description="Defines the number of different data sets in the ammo file", default=40000, required=True)

        with sdk2.parameters.Group('Parameters for clearing database') as comparing_block:
            clear_mailish = sdk2.parameters.String('Script for clearing', default='https://proxy.sandbox.yandex-team.ru/1406997193')
            pgpass = sdk2.parameters.String('File for PG auth', default='https://proxy.sandbox.yandex-team.ru/1407000736')
            ufile = sdk2.parameters.String('File with the uids', default='https://proxy.sandbox.yandex-team.ru/1406993877')

        with sdk2.parameters.Output:
            lunapark_job_id = sdk2.parameters.String('Lunapark job id', default_value='')
            lunapark_link = sdk2.parameters.String('Lunapark link', default_value='')

    def clear_mailish(self):
        script = get_source(self.Parameters.clear_mailish, 'clear_mailish')
        pgpass = get_source(self.Parameters.pgpass, '.pgpass')
        ufile = get_source(self.Parameters.ufile, 'uids')
        clear_command = "chmod +x {0} && export PGPASSFILE={1} && {0} {2}".format(script, pgpass, ufile)
        self.Context.clear_command = clear_command
        clear_process = subprocess.Popen(clear_command, bufsize=0, preexec_fn=os.setsid, close_fds=True, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        return clear_process.wait() == 0

    def start_shooting(self, desc, config_content):
        container = YANDEX_TANKAPI_LXC_CONTAINER.find(state=ctr.State.READY).order(-YANDEX_TANKAPI_LXC_CONTAINER.id).first().id
        if container is not None:
            self.Context.lxc_container = str(container)
        shooting_task = ShootViaTankapi(
            self,
            description=desc,
            ammo_source='in_config',
            config_source='file',
            config_content=config_content,
            monitoring_source='in_config',
            tanks=[self.Parameters.use_tank],
            container=container
            ).enqueue()
        self.loger.info('Subtask with shooting is started')
        self.Context.shooting_task_id = str(shooting_task.id)
        raise sdk2.WaitTask([shooting_task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=14400)

    def find_shooting(self):
        shooting_find = sdk2.Task.find(
            id=self.Context.shooting_task_id,
            children=True
        ).first()
        self.Context.shooting_find_id = str(shooting_find.id)
        self.loger.info('found ammo_generator_task_id = %s', shooting_find.id)
        self.Parameters.lunapark_job_id = shooting_find.Parameters.lunapark_job_id
        self.Parameters.lunapark_link = shooting_find.Parameters.lunapark_link

    def meta_parser(self):
        meta = []
        filename = "source"
        get_source(self.Parameters.users_file, filename)
        with open(filename, "r") as fd:
            for line in fd:
                try:
                    uid, email = line.strip().split(":")
                    meta.append((uid, email))
                except Exception as ex:
                    raise ce.TaskFailure("ERROR! Wrong file format. expected ':' as separator. {}".format(ex))
        return meta

    def ammo_data(self):
        i = 0
        bullits = []
        meta = self.meta_parser()
        length = len(meta)
        distr = [weighted_json_choice(self.Parameters.profile) for _ in xrange(self.Parameters.defcount)]
        for tag in distr:
            try:
                handler, size = tag.split("_")[:2]
            except ValueError:
                handler, size = tag.split("_", 1)[0], self.Parameters.defsize
            if handler in legalHandlers:
                bullit = {'tag': tag, 'uid': meta[i % length][0], 'email': meta[i % length][1], 'eml': size + '.eml'}
                bullits.append(bullit)
            else:
                raise ce.TaskFailure("WARNING! Invalid case {} in profile".format(tag))
            i += 1
        return ("\n".join(json.dumps(bullit) for bullit in bullits)).encode('utf-8')

    def make_ammo(self):
        ammo = AMMO_FILE(self, 'smtpgate ammo', 'ammo.json', ttl=1)
        ammoData = sdk2.ResourceData(ammo)
        try:
            ammoData.path.write_bytes(self.ammo_data())
            ammoData.ready()
            self.Context.ammo_resource = str(ammoData.path)
            return ammo
        except Exception as ex:
            raise ce.TaskFailure("Can not create ammo resource. Exception: {}".format(ex))

    def make_conf(self):
        autostop = []
        scheduler = []
        config_content = fu.read_file(get_source(self.Parameters.load_file, 'load.yaml'))
        yaml_config = yaml.safe_load(config_content)
        # Modify config for current task
        yaml_config['pandora']['pandora_cmd'] = self.Parameters.pandora.encode('utf-8')
        yaml_config['pandora']['config_content']['pools'][0]['id'] = "SMTPGate pool"
        yaml_config['pandora']['config_content']['pools'][0]['gun']['type'] = "smtpgate"
        yaml_config['pandora']['config_content']['pools'][0]['gun']['target'] = self.Parameters.host.encode('utf-8')
        yaml_config['pandora']['config_content']['pools'][0]['ammo']['type'] = "smtpgate_provider"
        yaml_config['pandora']['config_content']['pools'][0]['startup']['times'] = self.Parameters.instances
        # Ammo resource
        yaml_config['pandora']['resources'][0]['src'] = self.Context.ammo_url
        yaml_config['pandora']['resources'][0]['dst'] = './ammo.json'
        yaml_config['pandora']['resources'][1]['src'] = self.Parameters.monitoring_file.encode('utf-8')
        yaml_config['pandora']['resources'][1]['dst'] = './monitoring.xml'
        # Uploader section
        yaml_config['uploader']['api_address'] = "https://lunapark.yandex-team.ru/"
        yaml_config['uploader']['job_name'] = "[smtpgate][{}][{}]".format(self.Parameters.scheduler[0], self.Parameters.comment).encode('utf-8')
        yaml_config['uploader']['operator'] = "lunapark"
        yaml_config['uploader']['task'] = self.Parameters.ticket.encode('utf-8')

        for condition in self.Parameters.autostop:
            autostop.append(condition.encode('utf-8'))
        yaml_config['autostop']['autostop'] = autostop
        for rps in self.Parameters.scheduler:
            scheduler.append(ast.literal_eval(rps.encode('utf-8')))
        yaml_config['pandora']['config_content']['pools'][0]['rps'] = scheduler
        self.loger.info("scheduler is {}".format(yaml_config['pandora']['config_content']['pools'][0]['rps']))
        return yaml.dump(yaml_config, default_flow_style=False, encoding=None)

    def on_execute(self):
        self.loger = logger()
        self.loger.info("Start SMTPGate shooting")
        self.Context.shootype = str(json.loads(self.Parameters.scheduler[0])['type'])
        desc = self.Parameters.comment

        with self.memoize_stage.make_ammo:
            ammo_resource = self.make_ammo()
            self.Context.ammo_url = str(ammo_resource.http_proxy)

        with self.memoize_stage.make_conf:
            config_content = self.make_conf()

        with self.memoize_stage.shooting:
            self.start_shooting(desc, config_content)

        if self.Context.shooting_task_id != "":
            self.find_shooting()

        with self.memoize_stage.clearing:
            self.Context.clear_result = self.clear_mailish()

        if self.Parameters.st_token != "" and self.Context.send_comment_task_id == "":
            send_comment_task = LoadTestResults(
                self,
                shoot_id=self.Parameters.lunapark_job_id,
                report_type=self.Context.shootype,
                ticket_id=self.Parameters.ticket,
                st_token_name=self.Parameters.st_token,
                send_comment=True).enqueue()
            self.Context.send_comment_task_id = str(send_comment_task.id)
            raise sdk2.WaitTask([send_comment_task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=14400)


def logger():
    loggerr = logging.getLogger('%s_%s' % (__name__, time.time()))
    loggerr.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s %(levelname)s [%(processName)s: %(threadName)s] %(message)s')
    file_handler = logging.handlers.RotatingFileHandler(
        'smtpgate_shooting.log',
        maxBytes=(1024 * 1024),
        backupCount=5
    )
    file_handler.setLevel(logging.DEBUG)
    file_handler.setFormatter(formatter)
    loggerr.addHandler(file_handler)
    return loggerr


def get_source(url, dst):
    session = requests.session()
    try:
        with open(dst, 'wb') as resource:
            resource.write(session.get(url, stream=True).content)
        return os.path.abspath(dst)
    except Exception as ex:
        raise ce.TaskFailure("Can't download resource. {}".format(ex))
    finally:
        session.close()


def weighted_json_choice(choices):
    '''dict where key is choice and value probability'''
    total = sum(choices[choice] for choice in choices)
    r = random.uniform(0, total)
    upto = 0
    for choice in choices:
        if upto + choices[choice] >= r:
            return choice
        upto += choices[choice]
