# coding: utf-8
from sandbox import sdk2
from sandbox.common import errors as ce
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt
from sandbox.projects.common import file_utils as fu

from sandbox.projects.mail.Load.ShootingComparison import ShootingComparison
from sandbox.projects.mail.Load.SwatShooting.projects import PROJECTS
from sandbox.projects.tank.load_resources.resources import AMMO_FILE
from sandbox.projects.tank.load_resources.resources import YANDEX_TANKAPI_LXC_CONTAINER
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from sandbox.projects.tank.LoadTestResults import LoadTestResults

import logging
import time
import os
import os.path
import yaml
import requests
from collections import namedtuple


PROJECTS_DICT = {project.title: project for project in PROJECTS}


class COMPARISON_IDS_FILE(sdk2.Resource):
    """
        Resource with comparison shooting ids
    """
    comparison_label = sdk2.parameters.String('Comparison label')


class SwatShooting(sdk2.Task):
    """ Load test task for mail/swat services """


    class Context(sdk2.Task.Context):
        ammo_resource = ""
        ammo_resource_url = ""
        ammo_generator_task_id = ""
        ammo_generator_find_id = ""
        shooting_task_id = ""
        shooting_find_id = ""
        comparison_task_id = ""
        comparison_reference_list = ""
        comparison_reference_resource = ""
        comparison_reference_resource_url = ""
        send_comment_task_id = ""
        lxc_container = ""


    class Requirements(sdk2.Requirements):
        disk_space = 1024   # 1GiB on disk
        cores = 1           # exactly 1 core
        ram = 2048          # 8GiB or less


    class Parameters(sdk2.Task.Parameters):

        descriptions = {"load_file": u'Текст конфигурации стрельбы в формате yaml. Если не указать, возьмётся конфигурация по-умолчанию (из аркадии)',
                        "ammo_file": u'Файл с патронами для нагрузочного теста. Если не указать, то возьмётся файл из аркадии для соответствующего проекта',
                        "ammo_plain": u'Патроны as-is. Эта опция приоритетнее файла. Опционально',
        }

        with sdk2.parameters.Group('Shooting parameters') as tankapi_block:
            # task = sdk2.parameters.String('Lead ticket', default='MAILDLV-3170')
            with sdk2.parameters.String("Project", required=True,
                                        multiline=True) as target:
                for project in PROJECTS:
                    target.values[project.title] = project.title
            tank_url = sdk2.parameters.String('Tank API URL', default='tank-mailtestnets-1.tank-mailtestnets.tank.swat.mail.stable.qloud-d.yandex.net', required=True, description='Format: fdqn[:port]')
            scheduler = sdk2.parameters.String('Load profile', default='line(1, 200, 300s)', required=True)
            suffix = sdk2.parameters.String('Comment on the shooting', default='Sandbox shooting')
            autostop = sdk2.parameters.List('Autostop requirements', default=['quantile(95,150ms,10s)', 'negative_http(2xx,10%,10s)'])
            api_token = sdk2.parameters.String('Vault record name with startrek+arcanum token', required=True)
            ticket = sdk2.parameters.String('Ticket for comment', default='PDDMAIL-454')

        with sdk2.parameters.Group('Override load files') as sources_block:
            load_file_content = sdk2.parameters.String('Template shooting config', multiline=True, description=descriptions["load_file"].encode('utf-8'), default='')
            ammo_file_url = sdk2.parameters.String('Ammo file', description=descriptions["ammo_file"].encode('utf-8'), default='')
            ammo_plain = sdk2.parameters.String('Plain ammo', multiline=True, description=descriptions["ammo_plain"].encode('utf-8'), default='')

        with sdk2.parameters.Group('Parameters for comparing of the shooting') as comparing_block:
            comparing = sdk2.parameters.Bool('To comparing the shooting?', default=True)
            with comparing.value[True]:
                override_reference_shootings = sdk2.parameters.List('Override list of reference lunapark shooting ids (to compare with)', default=[])
                threshold = sdk2.parameters.Integer('Permissible deviation in %', default=5)
                # tracker = sdk2.parameters.String('Tickets tracker', default='https://st-api.yandex-team.ru', required=True)

        with sdk2.parameters.Output:
            lunapark_job_id = sdk2.parameters.String('Lunapark job id', default_value="")
            lunapark_link = sdk2.parameters.String('Lunapark link', default_value="")


    def start_shooting(self, desc, ammo_resource, config_content):
        tanks = []
        tanks.append(self.Parameters.tank_url)
        container = YANDEX_TANKAPI_LXC_CONTAINER.find(state = ctr.State.READY).order(-YANDEX_TANKAPI_LXC_CONTAINER.id).first().id
        self.Context.lxc_container = str(container)
        self.Context.save()
        shooting_task = ShootViaTankapi(
            self,
            description = desc,
            ammo_source = 'resource',
            ammo_resource = ammo_resource,
            config_source = 'file',
            config_content = config_content,
            tanks = tanks,
            container = container).enqueue()

        self.Context.shooting_task_id = str(shooting_task.id)
        self.loger.info('Subtask with shooting is started')
        raise sdk2.WaitTask([shooting_task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=14400)


    def find_shooting(self):
        shooting_find = sdk2.Task.find(
            id=self.Context.shooting_task_id,
            children=True
        ).first()
        self.Context.shooting_find_id = str(shooting_find.id)
        self.loger.info('found ammo_generator_task_id = %s', shooting_find.id)
        self.Parameters.lunapark_job_id = shooting_find.Parameters.lunapark_job_id
        self.Parameters.lunapark_link = shooting_find.Parameters.lunapark_link


    def get_target(self):
        target = self.Parameters.target
        try:
            return PROJECTS_DICT[target]
        except KeyError:
            raise ValueError('Unknown project {}'.format(target))


    def make_conf(self):
        autostop = []

        target = self.get_target()

        config_content = self.Parameters.load_file_content or fu.read_file(get_source(target.tank_config_url, 'load.yaml', token=self.Parameters.api_token))
        yaml_config = yaml.safe_load(config_content)

        yaml_config['phantom']['address'] = target.host.encode('utf-8')
        yaml_config['phantom']['headers'] = ['Host: {}'.format(target.host).encode('utf-8')]
        yaml_config['phantom']['load_profile']['schedule'] = self.Parameters.scheduler.encode('utf-8')
        yaml_config['yasm'].setdefault('panels', {})
        yaml_config['yasm']['panels'].setdefault('tank', {})
        yaml_config['yasm']['panels']['tank']['host'] = "QLOUD".encode('utf-8')
        yaml_config['yasm']['panels']['tank']['tags'] = "itype=qloud;prj=mail.swat.tank;tier={}".format(self.Parameters.tank_url.split('.')[0]).encode('utf-8')
        yaml_config['uploader']['job_name'] = "[mail-swat-{}][{}][{}]".format(target.title, self.Parameters.scheduler, self.Parameters.suffix).encode('utf-8')
        yaml_config['uploader'].setdefault('meta', {})
        yaml_config['uploader']['meta']['use_tank'] = self.Parameters.tank_url.encode('utf-8')
        yaml_config['uploader']['task'] = self.Parameters.ticket.encode('utf-8')
        for condition in self.Parameters.autostop:
            autostop.append(condition.encode('utf-8'))
        yaml_config.setdefault('autostop', {})
        yaml_config['autostop']['autostop'] = autostop

        return yaml.dump(yaml_config, default_flow_style=False, encoding=None)


    def comparing_shooting(self):
        target = self.get_target()
        reference_shootings_list = self.Parameters.override_reference_shootings or target.reference_shootings
        self.Context.comparison_reference_list = reference_shootings_list

        comparison = COMPARISON_IDS_FILE(self, 'My comparison', 'comparison.txt', ttl=1)
        comparison_data = sdk2.ResourceData(comparison)
        comparison_data.path.write_bytes('\n'.join(str(item) for item in reference_shootings_list).encode('utf-8'))
        comparison_data.ready()
        self.Context.comparison_reference_resource = str(comparison_data.path)

        comparison_data_url = str(comparison.http_proxy)
        self.Context.comparison_reference_resource_url = comparison_data_url

        comparison_task = ShootingComparison(
            self,
            sid = self.Parameters.lunapark_job_id,
            rfile = comparison_data_url).enqueue()

        self.Context.comparison_task_id = str(comparison_task.id)
        raise sdk2.WaitTask([comparison_task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=14400)


    def comparison_result(self):
        comparison_find = sdk2.Task.find(
            id=self.Context.comparison_task_id,
            children=True
        ).first()
        comparison_result = comparison_find.Parameters.comparison_result
        shooting_type = comparison_find.Parameters.shooting_type
# Build comment for ticket
        header = "Regression"
        footer = "https://wiki.yandex-team.ru/Load/"
        comment_for_ticket = "((@lix0 Load Support))"
        comparison_status = comparison_result.split("\n")[-1]
        comparison_body = "\n".join(comparison_result.split("\n")[:-1])
        comment_for_ticket += "\n<{%s\n#|\n||%s||\n|#\n%s}>\n%s"%(
            header,
            comparison_body.replace("\n","||\n||").replace("passed","!!(green)passed!!").replace("failed","!!(red)failed!!").replace("improved","!!(yellow)improved!!"),
            footer,
            comparison_status.replace("passed","**!!(green)SUCCESS!!**").replace("failed","**!!(red)FAILED!!**")
            )
        self.Context.comment_for_ticket = comment_for_ticket

        send_comment_task = LoadTestResults(
            self,
            shoot_id = self.Parameters.lunapark_job_id,
            report_type = shooting_type,
            additional_message = comment_for_ticket,
            ticket_id = self.Parameters.ticket,
            st_token_name = self.Parameters.api_token,
            send_comment = True).enqueue()

        self.Context.send_comment_task_id = str(send_comment_task.id)
        self.loger.info('Subtask with shooting is started')
        raise sdk2.WaitTask([send_comment_task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=14400)

    def get_ammo(self):
        ammo = AMMO_FILE(self, 'My ammo', 'ammo.txt', ttl=1)
        ammo_data = sdk2.ResourceData(ammo)

        if self.Parameters.ammo_plain:
            ammo_plain = self.Parameters.ammo_plain.replace('\n', '\r\n') + '\r\n\r\n'
            ammo_data.path.write_bytes(ammo_plain.encode('utf-8'))
        else:
            target = self.get_target()
            ammo_url = self.Parameters.ammo_file_url or target.ammo_url
            ammo_file = get_source(ammo_url, 'ammo', token=self.Parameters.api_token)

            with open(ammo_file, 'rb') as resource:
                ammo_data.path.write_bytes(resource.read())
        ammo_data.ready()
        self.Context.ammo_resource = str(ammo_data.path)
        return ammo

    def run_setup_hooks(self):
        target  = self.get_target()
        for hook in target.setup_hooks:
            hook()

    def on_execute(self):
        self.loger = logger()
        self.loger.info("Start mail/swat shooting")
        desc = self.Parameters.suffix

        self.run_setup_hooks()

        with self.memoize_stage.make_conf:
            config_content = self.make_conf()

        with self.memoize_stage.shooting:
            ammo_resource = self.get_ammo()
            self.Context.ammo_resource_url = str(ammo_resource.http_proxy)
            self.start_shooting(desc, ammo_resource, config_content)

        if self.Context.shooting_task_id != "":
            self.find_shooting()

# Evaluate shooting and send comment with the result of evaluation into ticket
        if self.Parameters.comparing and self.Parameters.lunapark_job_id != "":
            if self.Context.comparison_task_id == "":
                self.comparing_shooting()
            elif self.Context.send_comment_task_id == "":
                self.comparison_result()
            else:
                pass

        self.loger.info("Finish mail/swat shooting")

#  ========== End Of Class ==========  #

def logger():
    loggerr = logging.getLogger('%s_%s' % (__name__, time.time()))
    loggerr.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s %(levelname)s [%(processName)s: %(threadName)s] %(message)s')
    file_handler = logging.handlers.RotatingFileHandler(
        'swat_shooting.log',
        maxBytes = 1024 * 1024,
        backupCount = 5
    )

    file_handler.setLevel(logging.DEBUG)
    file_handler.setFormatter(formatter)
    loggerr.addHandler(file_handler)
    return loggerr


def get_source(url, dst, token):
    session = requests.session()
    try:
        oauth_token = sdk2.Vault.data(token)
        with open(dst, 'wb') as resource:
            response = session.get(url,
                                   headers={'Authorization': 'OAuth {}'.format(oauth_token)},
                                   stream=True)
            resource.write(response.content)
        return os.path.abspath(dst)
    except Exception as ex:
        raise ce.TaskFailure("Can't download resource. {}".format(ex))
    finally:
        session.close()
