from sandbox import sdk2
from sandbox.common import errors as ce
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt
from sandbox.projects.common import file_utils as fu

from sandbox.projects.mail.Load.WebApiAmmoGen import WebApiAmmoGen
from sandbox.projects.mail.Load.ShootingComparison import ShootingComparison
from sandbox.projects.tank.load_resources.resources import YANDEX_TANKAPI_LXC_CONTAINER
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from sandbox.projects.tank.LoadTestResults import LoadTestResults

import yaml
import logging
import os.path
import requests
import time
import ast
import os


DefaultProfile = {
    "messages@liza": 127,
    "do-client-log@journal": 122,
    "message-body@liza": 112,
    "system-status@liza": 80,
    "do-journal-log@journal": 70,
    "do-mail-reset-recent-counter@liza": 47,
    "message@liza": 40,
    "do-action-log@journal": 30,
    "recipients@liza": 19,
    "messages-pager@liza": 19,
    "message-nearest@liza": 15,
    "search-suggest@liza": 14,
    "reset-unvisited@liza": 13,
    "do-messages@liza": 12,
    "todo-timestamp@liza": 11,
    "email-info@liza": 11,
    "user-dropdown-data@liza": 8,
    "filters@liza": 8,
    "userphones@liza": 8,
    "collectors@liza": 8,
    "get_user_activity@liza": 8,
    "messages-types@liza": 7,
    "account-information@liza": 6,
    "do-service-worker-log@liza": 6,
    "map-data@liza": 2,
    "get-user-events@liza": 1,
    "captcha@liza": 1,
    "informer-weather-ng@liza": 1,
    "labels@liza": 1,
    "do-settings,settings@liza": 20,
    "folders,labels@liza": 50,
    "message-body,message-nearest@liza": 11,
    "push-subscriptions@liza": 2,
    "do-messages,folders,labels@liza": 6,
    "folders,labels,messages@liza": 7,
    "folders,labels,messages,message@liza": 2,
    "do-label@liza": 1,
    "folders@liza": 1,
    "message-in-reply-to@liza": 1,
    "message-in-reply-to-ng@liza": 1,
    "translate-langs@liza": 1,
    "unsubscribe-furita-newsletters@liza": 1,
    "ws-header-data@liza": 1,
    "do-settings@touch": 3,
    "avatars@touch": 2,
    "do-messages,folders@touch": 2,
    "message,message-body@touch": 2,
    "account-information,settings,folders,labels,multi-authorization-accounts,messages,user-activity,user-apps-activity@touch": 2,
    "do-search-suggest@touch": 1,
    "do-reset-unvisited@touch": 1,
    "do-folder-clear@touch": 1,
    "newsletter-counters@touch": 1,
    "mail-attach-resources@touch": 1,
    "labels,folders@touch": 1,
    "disk-resources-portion@touch": 1
}


MailingProfile = """{
    "get-newsletters@v1": 34,
    "get-newsletter-filters@v1": 35,
    "create-newsletter-filters@v1": 2,
    "delete-newsletter-filters@v1": 4,
    "get-messages@v1": 178,
    "get-emails-info@v1": 34,
    "get-ckey@v1": 80,
    "trash-messages@v1": 74,
    "get-folders@v1": 28
}"""


class WebApiShooting(sdk2.Task):

    class Context(sdk2.Task.Context):
        ammo_url = ""
        ammo_generator_task_id = ""
        ammo_generator_find_id = ""
        shooting_task_id = ""
        shooting_find_id = ""
        comparison_task_id = ""
        send_comment_task_id = ""
        lxc_container = ""

    class Requirements(sdk2.Requirements):
        disk_space = 1024   # 1GiB on disk
        cores = 1           # exactly 1 core
        ram = 2048          # 2GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Shooting parameters') as tankapi_block:
            comment = sdk2.parameters.String('Comment on the shooting', default='Sandbox shooting')
            target = sdk2.parameters.String('Target ES', default='mail_webapi_load.webapi', required=True)
            ssl = sdk2.parameters.Bool('SSL shooting', default=True)
            scheduler = sdk2.parameters.List('Load profile', default=['{"duration": "300s", "type": "const", "ops": 150}'])
            pandora = sdk2.parameters.String('Pandora binary link', default='https://proxy.sandbox.yandex-team.ru/2542903533', required=True)
            profile = sdk2.parameters.JSON('Shooting profile', default=DefaultProfile, description='For mailing shooting use profile: {}'.format(MailingProfile), required=True)
            autostop = sdk2.parameters.List('Autostop requirements', default=['quantile(90,500ms,20s)', 'negative_http(2xx,20%,20s)'])
            component = sdk2.parameters.String('Regression component', default='')
            instances = sdk2.parameters.Integer('Pandora instances', default=2000, required=True)
            pgtype = sdk2.parameters.String('Type of the pg users', default='type500')

        with sdk2.parameters.Group('Uploaded files') as sources_block:
            load_file = sdk2.parameters.String('Template shooting config', default='https://proxy.sandbox.yandex-team.ru/3192331267')
            users_file = sdk2.parameters.String('Resource for ammo generator', default='https://proxy.sandbox.yandex-team.ru/1210900337')

        with sdk2.parameters.Group('Experiments parameters') as experiments_block:
            use_experiments = sdk2.parameters.Bool('Add experiments values into shooting?')
            with use_experiments.value[True]:
                esource = sdk2.parameters.String('Source with the experiments', default='https://proxy.sandbox.yandex-team.ru/1229816161')
                eweight = sdk2.parameters.Integer('Percent of the experiments', default=0)

        with sdk2.parameters.Group('The choice of the tank') as tank_coice_block:
            with sdk2.parameters.RadioGroup('Use tank', required=True) as use_tank:
                use_tank.values['mimas.tanks.mail.yandex.net'] = use_tank.Value(value='MIMAS', default=True)
                use_tank.values['rhea.tanks.mail.yandex.net'] = use_tank.Value(value='RHEA')
                use_tank.values['iapet.tanks.mail.yandex.net'] = use_tank.Value(value='IAPET')

        with sdk2.parameters.Group('Parameters for comparing of the shooting') as comparing_block:
            comparing = sdk2.parameters.Bool('To comparing the shooting?', default=True)
            with comparing.value[True]:
                rfile = sdk2.parameters.String('File with the reference shootings', default='https://proxy.sandbox.yandex-team.ru/1268412278')
                threshold = sdk2.parameters.Integer('Permissible deviation in %', default=5)
                ticket = sdk2.parameters.String('Ticket for comment', default='MAILAPI-461')
                st_token = sdk2.parameters.String('Vault record name with startrek token', default='robot-mail-internal-startrek-token')

        with sdk2.parameters.Output:
            lunapark_job_id = sdk2.parameters.String('Lunapark job id', default_value='')
            lunapark_link = sdk2.parameters.String('Lunapark link', default_value='')

    def start_shooting(self, desc, config_content):
        tanks = []
        tanks.append(self.Parameters.use_tank)
        container = YANDEX_TANKAPI_LXC_CONTAINER.find(state=ctr.State.READY).order(-YANDEX_TANKAPI_LXC_CONTAINER.id).first().id
        if container is not None:
            self.Context.lxc_container = str(container)
        shooting_task = ShootViaTankapi(
            self,
            description=desc,
            ammo_source='in_config',
            config_source='file',
            config_content=config_content,
            tanks=tanks,
            container=container
            ).enqueue()
        self.loger.info('Subtask with shooting is started')
        self.Context.shooting_task_id = str(shooting_task.id)
        raise sdk2.WaitTask([shooting_task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=14400)

    def find_shooting(self):
        shooting_find = sdk2.Task.find(
            id=self.Context.shooting_task_id,
            children=True
        ).first()
        self.Context.shooting_find_id = str(shooting_find.id)
        self.loger.info('found ammo_generator_task_id = %s', shooting_find.id)
        self.Parameters.lunapark_job_id = shooting_find.Parameters.lunapark_job_id
        self.Parameters.lunapark_link = shooting_find.Parameters.lunapark_link

    def make_ammo(self):
        self.loger.info('Start ammo generation')
        esource = ""
        eweight = 0
        if self.Parameters.use_experiments:
            esource = self.Parameters.esource
            eweight = self.Parameters.eweight
        ammo_gen_task = WebApiAmmoGen(
            self,
            description='ammo generator for task {0}'.format(self.id),
            wmi=self.es_to_fqdn(self.Parameters.target),
            source=self.Parameters.users_file,
            profile=self.Parameters.profile,
            pgtype=self.Parameters.pgtype,
            esource=esource,
            eweight=eweight
            ).enqueue()
        self.Context.ammo_generator_task_id = str(ammo_gen_task.id)
        self.Context.save()
        raise sdk2.WaitOutput({ammo_gen_task.id: 'ammo'}, wait_all=True, timeout=3600)

    def get_ammo(self):
        ammo_gen_find = self.find(id=self.Context.ammo_generator_task_id).first()
        self.Context.ammo_generator_find_id = str(ammo_gen_find.id)
        self.loger.info('found ammo_generator_task_id = %s', ammo_gen_find.id)
        return ammo_gen_find.Parameters.ammo

    def make_conf(self):
        autostop = []
        scheduler = []
        config_content = fu.read_file(get_source(self.Parameters.load_file, 'load.yaml'))
        yaml_config = yaml.safe_load(config_content)
        # Modify config for current task
        yaml_config['pandora']['pandora_cmd'] = self.Parameters.pandora.encode('utf-8')
        yaml_config['pandora']['config_content']['pools'][0]['gun']['target'] = ':'.join([self.es_to_fqdn(self.Parameters.target).encode('utf-8'), '443' if self.Parameters.ssl else '80'])
        yaml_config['pandora']['config_content']['pools'][0]['gun']['ssl'] = self.Parameters.ssl
        yaml_config['pandora']['config_content']['pools'][0]['startup']['times'] = self.Parameters.instances
        yaml_config['pandora']['resources'][0]['src'] = self.Context.ammo_url
        yaml_config['pandora']['resources'][0]['dst'] = './ammo.json'
        yaml_config['yasm']['panels']['web-api']['tags'] = "itype=mail;stage=mail_webapi_load;deploy_unit={}".format(self.Parameters.target.split('.')[1]).encode('utf-8')
        yaml_config['yasm']['panels']['tank']['tags'] = "itype=mail_load;stage=mail_load_tanks;deploy_unit={}".format(self.Parameters.use_tank.split('.')[0]).encode('utf-8')
        yaml_config['uploader']['job_name'] = '[web-api][{}][{}]'.format(self.Parameters.scheduler, self.Parameters.comment).encode('utf-8')

        if self.Parameters.component:
            yaml_config['uploader']['component'] = self.Parameters.component.encode('utf-8')

        for condition in self.Parameters.autostop:
            autostop.append(condition.encode('utf-8'))
        yaml_config['autostop']['autostop'] = autostop

        for rps in self.Parameters.scheduler:
            scheduler.append(ast.literal_eval(rps.encode('utf-8')))
        yaml_config['pandora']['config_content']['pools'][0]['rps'] = scheduler
        self.loger.info('scheduler is {}'.format(yaml_config['pandora']['config_content']['pools'][0]['rps']))

        return yaml.dump(yaml_config, default_flow_style=False, encoding=None)

    def comparing_shooting(self):
        comparison_task = ShootingComparison(
            self,
            sid=self.Parameters.lunapark_job_id,
            rfile=self.Parameters.rfile).enqueue()
        self.Context.comparison_task_id = str(comparison_task.id)
        raise sdk2.WaitTask([comparison_task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=14400)

    def comparison_result(self):
        # Search task with comparison of self shooting
        comparison_find = sdk2.Task.find(
            id=self.Context.comparison_task_id,
            children=True).first()
        # Get results of the comparison
        comparison_result = comparison_find.Parameters.comparison_result
        shooting_type = comparison_find.Parameters.shooting_type
        # Build comment for ticket
        header = "Regression"
        footer = "https://wiki.yandex-team.ru/Load/"
        comment_for_ticket = "((@lix0 Load Support))"
        comparison_status = comparison_result.split("\n")[-1]
        comparison_body = "\n".join(comparison_result.split("\n")[:-1])
        comment_for_ticket += "\n<{%s\n#|\n||%s||\n|#\n%s>\n%s" % (
            header,
            comparison_body.replace("\n", "||\n||").replace("passed", "!!(green)passed!!").replace("failed", "!!(red)failed!!").replace("improved", "!!(yellow)improved!!"),
            footer,
            comparison_status.replace("passed", "**!!(green)SUCCESS!!**").replace("failed", "**!!(red)FAILED!!**"))
        self.Context.comment_for_ticket = comment_for_ticket
        # Send comment into ticket
        send_comment_task = LoadTestResults(
            self,
            shoot_id=self.Parameters.lunapark_job_id,
            report_type=shooting_type,
            additional_message=comment_for_ticket,
            ticket_id=self.Parameters.ticket,
            st_token_name=self.Parameters.st_token,
            send_comment=True).enqueue()

        self.Context.send_comment_task_id = str(send_comment_task.id)
        self.loger.info('Subtask with shooting is started')
        raise sdk2.WaitTask([send_comment_task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=14400)

    def on_execute(self):
        self.loger = logger()
        self.loger.info("Start Web-Api shooting")

        desc = self.Parameters.comment

        with self.memoize_stage.make_ammo:
            self.make_ammo()

        with self.memoize_stage.make_conf:
            ammo_resource = self.get_ammo()
            self.Context.ammo_url = str(ammo_resource.http_proxy)
            config_content = self.make_conf()

        with self.memoize_stage.shooting:
            self.start_shooting(desc, config_content)

        if self.Context.shooting_task_id != "":
            self.find_shooting()
        # Evaluate shooting and send comment with the result of evaluation into ticket
        if self.Parameters.comparing and self.Parameters.lunapark_job_id != "":
            if self.Context.comparison_task_id == "":
                self.comparing_shooting()
            elif self.Context.send_comment_task_id == "":
                self.comparison_result()
            # ==== End Of Class ==== #

    def es_to_fqdn(self, endpoint_set_id):
        dcs = ['iva', 'myt', 'sas', 'vla']
        fqdns = []

        for dc in dcs:
            res = requests.get('http://sd.yandex.net:8080/resolve_endpoints/json', json={
                'cluster_name': dc,
                'endpoint_set_id': endpoint_set_id,
                'client_name': 'mail_load',
            })
            print(res.json())
            if 'endpoint_set' in res.json():
                if 'endpoints' in res.json()['endpoint_set']:
                    for endpoint in res.json()['endpoint_set']['endpoints']:
                        fqdns.append(endpoint['fqdn'])

        if len(fqdns) == 0:
            raise ce.TaskFailure(
                "[ERROR] Cannot get endpoints for endpoint_set {es}".format(es=endpoint_set_id))

        return fqdns[0]


def logger():
    loggerr = logging.getLogger('%s_%s' % (__name__, time.time()))
    loggerr.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s %(levelname)s [%(processName)s: %(threadName)s] %(message)s')
    file_handler = logging.handlers.RotatingFileHandler(
        'webapi_shooting.log',
        maxBytes=(1024 * 1024),
        backupCount=5)
    file_handler.setLevel(logging.DEBUG)
    file_handler.setFormatter(formatter)
    loggerr.addHandler(file_handler)
    return loggerr


def get_source(url, dst):
    session = requests.session()
    try:
        with open(dst, 'wb') as resource:
            resource.write(session.get(url, stream=True).content)
        return os.path.abspath(dst)
    except Exception as ex:
        raise ce.TaskFailure("Can't download resource. {}".format(ex))
    finally:
        session.close()
