# -*- coding: utf-8 -*-
import logging
import os
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.common.arcadia import sdk as arcadiasdk


class MAILFRONT_NODE_MODULES(sdk2.Resource):
    __default_attribute__ = sdk2.parameters.String

    releasable = True
    any_arch = True
    executable = False

    package = sdk2.Attributes.String('Package name', required=True)
    location = sdk2.Attributes.String('Package location', required=True)


class MailfrontNodeModulesBackup(sdk2.Task):
    name = 'MAILFRONT_NODE_MODULES_BACKUP'

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        container_resource = 2967194243
        dns = ctm.DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        client_tags = sdk2.parameters.ClientTags(
            "Client tags. May only narrow the default set of client_tags that is specified in the RunCommand.Requirements",
            default=ctc.Tag.LINUX_FOCAL
        )

        branch = sdk2.parameters.String(
            'Arcadia branch',
            description='Checkout branch',
            default='trunk',
            required=True,
        )
        project_path_in_arcadia = sdk2.parameters.String(
            'Path to project in Arcadia',
            description='Relative path',
            default='yandex360/frontend',
            required=True,
        )
        nodejs_version = sdk2.parameters.String(
            'Nodejs version',
            default='v14.16.1',
            required=True
        )
        command = sdk2.parameters.String(
            'Command to execute',
            default='npm --silent ci && node tools/ci/node-modules-backup.js "$(npx --no-install lerna ls --json)"',
            required=True
        )
        arc_token = sdk2.parameters.YavSecret(
            'Arc OAuth token',
            default='',
            required=True
        )
        ya_token = sdk2.parameters.YavSecret(
            'Ya token',
            default='',
            required=True
        )

    def _execute(self, command, shell=False):
        logging.debug('Try to call command: %s in path %s' % (str(command), os.getcwd()))
        result, err = sp.Popen(
            command, cwd=os.getcwd(), stdout=sp.PIPE, stderr=sp.PIPE, shell=shell
        ).communicate()

        logging.info('result')
        logging.info(result)
        logging.error('err')
        logging.error(err)

        return result

    def on_save(self):
        self.Requirements.client_tags = self.Parameters.client_tags

    def on_execute(self):
        branch = self.Parameters.branch

        logging.debug('Initializing arc branch %s' % branch)

        with arcadiasdk.mount_arc_path('arcadia-arc:/#trunk', arc_oauth_token=self.Parameters.arc_token.value()) as arcadia_path:
            arcadia_path = sdk2.Path(arcadia_path)
            logging.debug('Arcadia path: %s' % (str(arcadia_path)))
            project_path_in_arcadia = self.Parameters.project_path_in_arcadia

            with sdk2.helpers.ProcessLog(self, logger="command") as pl:
                commands = [
                    'source "$NVM_DIR/nvm.sh"',
                    'nvm use {}'.format(self.Parameters.nodejs_version),
                    self.Parameters.command
                ]

                sdk2.helpers.subprocess.check_call(
                    ' && '.join(commands),
                    shell=True,
                    stdout=pl.stdout,
                    stderr=pl.stderr,
                    close_fds=True,
                    env={
                        'YA_TOKEN': self.Parameters.ya_token.value(),
                        'NVM_DIR': '/opt/nvm',
                    },
                    cwd=str(arcadia_path / project_path_in_arcadia)
                )
