import logging
import json

from sandbox import sdk2

# from sandbox.projects.mail.CommonLib.lib.resources import MAIL_SHOOTING_MONITORING_INFO, MAIL_SHOOTING_AGGREGATE_INFO

OK_RESULT = 'ok'
FAIL_RESULT = 'fail'


class PerformanceCompareTask(sdk2.Task):

    class Context(sdk2.Context):
        compare_http_codes_distribution_result = None
        compare_net_codes_distribution_result = None
        compare_98_percentile_result = None
        compare_95_percentile_result = None
        compare_memory_usage_avg_result = None
        compare_memory_usage_max_result = None
        compare_user_cpu_usage_avg_result = None
        compare_user_cpu_usage_max_result = None

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 30*60
        owner = "MAIL"
        description = "Task for compare current service performance with standard performance"

        with sdk2.parameters.Group("Resources for compare") as staff_for_diff_block:
            monitoring_baseline = sdk2.parameters.Resource(
                'Base shooting monitoring metrics for service',
                # type=MAIL_SHOOTING_MONITORING_INFO,  # FIXME: invalid argument (SANDBOX-6404)
                required=True
                )

            monitoring_to_compare = sdk2.parameters.Resource(
                'Monitoring info for compare with baseline monitoring',
                # type=MAIL_SHOOTING_MONITORING_INFO,  # FIXME: invalid argument (SANDBOX-6404)
                required=True
                )

            common_info_baseline = sdk2.parameters.Resource(
                'Base shooting common metrics for service',
                # type=MAIL_SHOOTING_AGGREGATE_INFO,  # FIXME: invalid argument (SANDBOX-6404)
                required=True
                )

            common_info_to_compare = sdk2.parameters.Resource(
                'Common shooting metrics for compare with baseline',
                # type=MAIL_SHOOTING_AGGREGATE_INFO,  # FIXME: invalid argument (SANDBOX-6404)
                required=True
                )

        with sdk2.parameters.Group('Common metrics') as common_metrics_block:
            with sdk2.parameters.Group("Http codes metric") as http_codes_block:
                compare_http_codes_distribution = sdk2.parameters.Bool(
                    'Is compare http codes distribution',
                    default=True,
                    required=True)

                max_diff_percent_http_codes = sdk2.parameters.Float(
                    'Max percents difference for http codes',
                    default=1,
                    required=True)

            with sdk2.parameters.Group('Net codes metric') as net_codes_block:
                compare_net_codes_distribution = sdk2.parameters.Bool(
                    'Is compare net codes distribution',
                    default=False,
                    required=True)

                max_diff_percent_net_codes = sdk2.parameters.Float(
                    'Max percents difference for net codes',
                    default=2,
                    required=True)

            with sdk2.parameters.Group('Percentiles') as percentiles_block:
                compare_98_percentile_value = sdk2.parameters.Bool(
                    'Is compare 98 percentile values',
                    default=False,
                    required=True)

                max_diff_98_percentile_percents = sdk2.parameters.Float(
                    'Max difference in percents for 98 percentile',
                    default=10,
                    required=True)

                compare_95_percentile_value = sdk2.parameters.Bool(
                    'Is compare 95 percentile values',
                    default=False,
                    required=True)

                max_diff_95_percentile_percents = sdk2.parameters.Float(
                    'Max difference in percents for 95 percentile',
                    default=10,
                    required=True)

        with sdk2.parameters.Group('Monitoring metrics') as monitoring_metrics_block:
            with sdk2.parameters.Group('User memory usage') as user_memory_block:
                compare_memory_usage = sdk2.parameters.Bool(
                    'Is compare user memory usage',
                    default=False,
                    required=True
                )

                max_diff_memory_usage_avg_metric = sdk2.parameters.Float(
                   'Max difference between avg user memory usage',
                    default=7,
                    required=True
                )

                max_diff_memory_usage_max = sdk2.parameters.Float(
                    'Max difference between max user memory usage',
                    default=10,
                    required=True
                )

            with sdk2.parameters.Group('User cpu usage') as user_cpu_block:
                compare_user_cpu_usage = sdk2.parameters.Bool(
                    'Is compare user cpu usage',
                    default=False,
                    required=True
                )

                max_diff_user_cpu_usage_avg_metric = sdk2.parameters.Float(
                   'Max difference between avg user cpu usage',
                    default=7,
                    required=True
                )

                max_diff_user_cpu_usage_max = sdk2.parameters.Float(
                    'Max difference between max user cpu usage',
                    default=10,
                    required=True
                )

    def __check_all_rps_consists_in_common_reports(self):
        return all(map(lambda x: x in self.common_info_baseline, self.common_info_to_compare))

    def __check_all_rps_consists_in_monitoring_reports(self):
        return all(map(lambda x: x in self.baseline_monitoring, self.monitoring_to_compare))

    def __load_all_reports(self):
        baseline_monitoring = sdk2.resource.ResourceData(self.Parameters.monitoring_baseline).path.read_bytes()
        self.baseline_monitoring = json.loads(baseline_monitoring)

        monitoring_to_compare = sdk2.resource.ResourceData(self.Parameters.monitoring_to_compare).path.read_bytes()
        self.monitoring_to_compare = json.loads(monitoring_to_compare)

        common_info_baseline = sdk2.resource.ResourceData(self.Parameters.common_info_baseline).path.read_bytes()
        self.common_info_baseline = json.loads(common_info_baseline)

        common_info_to_compare = sdk2.resource.ResourceData(self.Parameters.common_info_to_compare).path.read_bytes()
        self.common_info_to_compare = json.loads(common_info_to_compare)
        logging.info("All reports load success!")

    def __create_log_info_str(self, baseline, to_compare, threshold, rps):
        return 'baseline:{} to compare:{} threshold:{} rps:{}'.format(
                str(baseline), str(to_compare), str(threshold), str(rps))

    def __compare_http_codes_distribution(self):
        for rps_value in self.common_info_to_compare:
            http_success_dist_baseline = filter(lambda x: x['http'] == 200, self.common_info_baseline[rps_value]['http'])[0]['percent']
            http_success_dist_to_compare = filter(lambda x: x['http'] == 200, self.common_info_to_compare[rps_value]['http'])[0]['percent']
            log_info = self.__create_log_info_str(
                http_success_dist_baseline, http_success_dist_to_compare, self.Parameters.max_diff_percent_http_codes, rps_value)
            if http_success_dist_baseline - http_success_dist_to_compare > self.Parameters.max_diff_percent_http_codes:
                self.Context.compare_http_codes_distribution_result = FAIL_RESULT
                self._failed_metrics.append('http codes distribution')
                logging.info('Compare http codes dist result: fail {}'.format(log_info))
                return 1

        self.Context.compare_http_codes_distribution_result = OK_RESULT
        logging.info('Compare http codes dist result: ok')
        return 0

    def __compare_net_codes_distribution(self):
        for rps_value in self.common_info_to_compare:
            net_success_dist_baseline = filter(lambda x: x['net'] == 0, self.common_info_baseline[rps_value]['net'])[0]['percent']
            net_success_dist_to_compare = filter(lambda x: x['net'] == 0, self.common_info_to_compare[rps_value]['net'])[0]['percent']
            log_info = self.__create_log_info_str(
                net_success_dist_baseline, net_success_dist_to_compare, self.Parameters.max_diff_percent_net_codes, rps_value)
            if net_success_dist_baseline - net_success_dist_to_compare > self.Parameters.max_diff_percent_net_codes:
                self.Context.compare_net_codes_distribution_result = FAIL_RESULT
                self._failed_metrics.append('net codes distribution')
                logging.info('Compare net codes dist result: fail {}'.format(log_info))
                return 1

        self.Context.compare_net_codes_distribution_result = OK_RESULT
        logging.info('Compare net codes dist result: ok')
        return 0

    def __compare_98_percentile_value(self):
        for rps_value in self.common_info_to_compare:
            percentile98_baseline = filter(lambda x: x['percentile'] == '98', self.common_info_baseline[rps_value]['percentiles'])[0]['ms']
            percentile98_to_compare = filter(lambda x: x['percentile'] == '98', self.common_info_to_compare[rps_value]['percentiles'])[0]['ms']
            log_info = self.__create_log_info_str(
                percentile98_baseline, percentile98_to_compare, self.Parameters.max_diff_98_percentile_percents, rps_value)
            if (percentile98_to_compare * 1. / percentile98_baseline - 1.) * 100. > self.Parameters.max_diff_98_percentile_percents:
                self.Context.compare_98_percentile_result = FAIL_RESULT
                self._failed_metrics.append('98 percentile')
                logging.info('Compare 98 percentile result: fail {}'.format(log_info))
                return 1

        self.Context.compare_98_percentile_result = OK_RESULT
        logging.info('Compare 98 percentile result: ok')
        return 0

    def __compare_95_percentile_value(self):
        for rps_value in self.common_info_to_compare:
            percentile95_baseline = filter(lambda x: x['percentile'] == '95', self.common_info_baseline[rps_value]['percentiles'])[0]['ms']
            percentile95_to_compare = filter(lambda x: x['percentile'] == '95', self.common_info_to_compare[rps_value]['percentiles'])[0]['ms']
            log_info = self.__create_log_info_str(
                percentile95_baseline, percentile95_to_compare, self.Parameters.max_diff_95_percentile_percents, rps_value)
            if (percentile95_to_compare * 1. / percentile95_baseline - 1.) * 100. > self.Parameters.max_diff_95_percentile_percents:
                self.Context.compare_95_percentile_result = FAIL_RESULT
                self._failed_metrics.append('95 percentile')
                logging.info('Compare 95 percentile result: fail {}'.format(log_info))
                return 1

        self.Context.compare_95_percentile_result = OK_RESULT
        logging.info('Compare 95 percentile result: ok')
        return 0

    def __compare_memory_usage(self):
        for rps_value in self.monitoring_to_compare:
            mem_usage_avg_baseline = self.baseline_monitoring[rps_value]['mem_usage']['avg']
            mem_usage_avg_to_compare = self.monitoring_to_compare[rps_value]['mem_usage']['avg']
            log_info_avg = self.__create_log_info_str(
                mem_usage_avg_baseline, mem_usage_avg_to_compare, self.Parameters.max_diff_memory_usage_avg_metric, rps_value)

            if (mem_usage_avg_to_compare * 1. / mem_usage_avg_baseline - 1.) * 100. > self.Parameters.max_diff_memory_usage_avg_metric:
                self.Context.compare_memory_usage_avg_result = FAIL_RESULT
                if 'memory usage avg' not in self._failed_metrics:
                    self._failed_metrics.append('memory usage avg')
                logging.info('Compare memory usage avg metric result: fail {}'.format(log_info_avg))

            mem_usage_max_baseline = self.baseline_monitoring[rps_value]['mem_usage']['max']
            mem_usage_max_to_compare = self.monitoring_to_compare[rps_value]['mem_usage']['max']
            log_info_max = self.__create_log_info_str(
                mem_usage_max_baseline, mem_usage_max_to_compare, self.Parameters.max_diff_memory_usage_max, rps_value)

            logging.debug(log_info_avg + ' ' + log_info_max)

            if (mem_usage_max_to_compare * 1. / mem_usage_max_baseline - 1.) * 100. > self.Parameters.max_diff_memory_usage_max:
                self.Context.compare_memory_usage_max_result = FAIL_RESULT
                if 'memory usage max' not in self._failed_metrics:
                    self._failed_metrics.append('memory usage max')
                logging.info('Compare memory usage max metric result: fail {}'.format(log_info_max))

        if self.Context.compare_memory_usage_avg_result != FAIL_RESULT:
            self.Context.compare_memory_usage_avg_result = OK_RESULT
            logging.info('Compare memory usage avg metric result: ok')
        if self.Context.compare_memory_usage_max_result != FAIL_RESULT:
            self.Context.compare_memory_usage_max_result = OK_RESULT
            logging.info('Compare memory usage max metric result: ok')
        return 0

    def __compare_user_cpu_usage(self):
        for rps_value in self.monitoring_to_compare:
            cpu_user_usage_avg_baseline = self.baseline_monitoring[rps_value]['cpu_user']['avg']
            cpu_user_usage_avg_to_compare = self.monitoring_to_compare[rps_value]['cpu_user']['avg']
            log_info_avg = self.__create_log_info_str(
                cpu_user_usage_avg_baseline, cpu_user_usage_avg_to_compare, self.Parameters.max_diff_user_cpu_usage_avg_metric, rps_value)

            if (cpu_user_usage_avg_to_compare * 1. / cpu_user_usage_avg_baseline - 1.) * 100. > self.Parameters.max_diff_user_cpu_usage_avg_metric:
                self.Context.compare_user_cpu_usage_avg_result = FAIL_RESULT
                if 'cpu usage avg' not in self._failed_metrics:
                    self._failed_metrics.append('cpu usage avg')
                logging.info('Compare user cpu usage avg metric result: fail {}'.format(log_info_avg))

            cpu_user_usage_max_baseline = self.baseline_monitoring[rps_value]['cpu_user']['max']
            cpu_user_usage_max_to_compare = self.monitoring_to_compare[rps_value]['cpu_user']['max']
            log_info_max = self.__create_log_info_str(
                cpu_user_usage_max_baseline, cpu_user_usage_max_to_compare, self.Parameters.max_diff_user_cpu_usage_max, rps_value)

            if (cpu_user_usage_max_to_compare * 1. / cpu_user_usage_max_baseline - 1.) * 100. > self.Parameters.max_diff_user_cpu_usage_max:
                self.Context.compare_user_cpu_usage_max_result = FAIL_RESULT
                if 'cpu usage max' not in self._failed_metrics:
                    self._failed_metrics.append('cpu usage max')
                logging.info('Compare user cpu usage max metric result: fail {}'.format(log_info_max))

        if self.Context.compare_user_cpu_usage_avg_result != FAIL_RESULT:
            self.Context.compare_user_cpu_usage_avg_result = OK_RESULT
            logging.info('Compare cpu user usage avg metric result: ok')
        if self.Context.compare_user_cpu_usage_max_result != FAIL_RESULT:
            self.Context.compare_user_cpu_usage_max_result = OK_RESULT
            logging.info('Compare cpu user usage max metric result: ok')

        return 0

    def __compare_metrics(self):
        if self.Parameters.compare_http_codes_distribution:
            self.__compare_http_codes_distribution()

        if self.Parameters.compare_net_codes_distribution:
            self.__compare_net_codes_distribution()

        if self.Parameters.compare_98_percentile_value:
            self.__compare_98_percentile_value()

        if self.Parameters.compare_95_percentile_value:
            self.__compare_95_percentile_value()

        if self.Parameters.compare_memory_usage:
            self.__compare_memory_usage()

        if self.Parameters.compare_user_cpu_usage:
            self.__compare_user_cpu_usage()

        logging.info('Compare metrics finished')

        error_msg = 'Compare metrics failed, metrics failed: '
        if len(self._failed_metrics):
            error_msg = 'Compare metrics failed, metrics failed: ' + str(self._failed_metrics)
            self.set_info(error_msg)
            raise Exception(error_msg)
        return 0

    def on_execute(self):
        self.__load_all_reports()

        if self.Parameters.compare_memory_usage or \
            self.Parameters.compare_user_cpu_usage:
            assert(self.__check_all_rps_consists_in_monitoring_reports())

        if self.Parameters.compare_http_codes_distribution or \
            self.Parameters.compare_net_codes_distribution:
            assert(self.__check_all_rps_consists_in_common_reports())

        self._failed_metrics = []

        self.__compare_metrics()
        return 0
