# coding=utf-8
import logging
from sandbox import sdk2
import sandbox.common.types.task as ctt

from sandbox.common.errors import TaskFailure
from sandbox.projects.mail.CommonLib.lib.aqua import Aqua
from sandbox.projects.mail.CommonLib.lib.star_track import StarTrackApi


class RunAquaTests(sdk2.Task):
    """Run aqua tests"""

    class Context(sdk2.Context):
        launch_id = None
        launch_info = None

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 30*60
        owner = "MAIL"
        description = "Run aqua tests"
        priority = ctt.Priority(ctt.Priority.Class.USER, ctt.Priority.Subclass.NORMAL)
        pack_name = sdk2.parameters.String("Aqua test pack name")
        poll_interval = sdk2.parameters.Integer("Tests status poll interval (seconds)", default=10)
        timeout = sdk2.parameters.Integer("Tests run timeout (minutes)", default=30)
        tag = sdk2.parameters.String("Tag")
        threshold = sdk2.parameters.Integer("Tests revoke threshold", default=60)
        props = sdk2.parameters.Dict("Launch properties")
        with sdk2.parameters.Group("StarTrack parameters (optional)") as star_track_block:
            issues = sdk2.parameters.List("Issues id")
            st_token_vault_item = sdk2.parameters.String("Vault item containing ST OAuth token",
                                                         default='robot-gerrit-oauth-startrek')

    def _make_result_message(self):
        launch_info = self.Context.launch_info
        return "Задачи: {}\n".format(' '.join(self.Parameters.issues)) +\
               "Тесты завершились " +\
               "(({} {})) - OK: !!(green)**{}**!!, ПОТРАЧЕНО: !!**{}**!!, ВСЕГО: **{}**".format(
                   launch_info['reportRequestUrl'],
                   self.Parameters.pack_name.encode('utf-8'),
                   launch_info['passedSuites'],
                   launch_info['failedSuites'],
                   launch_info['totalSuites']
               )

    def _make_fail_message(self):
        return "Задачи: {}\nТесты сломаны".format(' '.join(self.Parameters.issues))

    def _post_message(self, message):
        issues = self.Parameters.issues
        if issues:
            api = StarTrackApi(sdk2.Vault.data(self.Parameters.st_token_vault_item))
            errors = api.add_comments(issues, text=message, task_id=self.id)
            if errors:
                logging.error("Errors while commenting star track: {}".format(errors))

    def on_execute(self):
        try:
            pack_name = self.Parameters.pack_name
            logging.info("Starting AQUA test pack '{}'".format(pack_name.encode('utf-8')))
            aqua_api = Aqua(pack_name)
            predefined_params = {
                'should.be.successful.in.aqua.plugin.percents': str(self.Parameters.threshold),
                'set.tag.for.launch.by.aqua.plugin': self.Parameters.tag
            }
            pack_params = dict(self.Parameters.props.items() + predefined_params.items())
            self.Context.launch_id = aqua_api.launch_template(pack_params)
            logging.info("Test pack is started with launch id {}".format(self.Context.launch_id))
            self.Context.launch_info = aqua_api.wait_launch(self.Context.launch_id, self.Parameters.poll_interval,
                                                            self.Parameters.timeout * 60)
            logging.info("Tests pack launch '{}' is complete".format(self.Context.launch_id))
        except Exception as e:
            raise TaskFailure(e)

    # noinspection PyUnusedLocal
    def on_success(self, prev_status):
        self._post_message(self._make_result_message())

    # noinspection PyUnusedLocal
    def on_failure(self, prev_status):
        self._post_message(self._make_fail_message())
