import logging
from datetime import datetime, timedelta
import time

from requests.auth import HTTPBasicAuth
import task_api

logger = logging.getLogger(__name__)


class JenkinsClientError(RuntimeError):
    def __init__(self, message):
        super(RuntimeError, self).__init__(message)


class JenkinsFailedJobError(RuntimeError):
    def __init__(self, message):
        super(RuntimeError, self).__init__(message)


class JenkinsClient(object):
    class _JobHandler(object):
        """
        .../job/build gives reference to queue item which is not running job.
        We have to poll state of this queue item to get running job state.

        But we want to have some simple interface like 'client.run_job(...).await()'.
        Lets hide this indirection inside the class, so it will await both queue extraction and job execution
        """

        _TERMINAL_STATES = {"SUCCESS", "FAILURE"}

        def __init__(self, client, job_name, queue_item_id=None, job_id=None):
            self._client = client
            self.job_name = job_name

            if job_id is None and queue_item_id is None:
                raise JenkinsClientError("Cannot create JobHandler without any of id: queue_item_id and job_id")

            self.job_id = job_id  # May be None if job is still scheduled in queue
            self.queue_item_id = queue_item_id  # May be None if we know existing job_id

            self._queue_item = None
            self._job = None

        def _update(self):
            if self.job_id is not None:
                self._update_job()
            elif self.queue_item_id is not None:
                self._update_queue_item()
            else:
                raise JenkinsClientError("Cannot update job state: no ids are present")

        def _update_queue_item(self):
            if self.queue_item_id is None:
                raise JenkinsClientError("Cannot update queue item without queue_item_id")

            self._queue_item = self._client._call_with_host_and_auth(method=task_api.get_queue_item, queue_item_id=self.queue_item_id)
            if self.job_id is None:
                self.job_id = self._extract_job_id_from_queue_item(self._queue_item)

        def _update_job(self):
            if self.job_id is None:
                raise JenkinsClientError("Cannot update job info without job_id")

            self._job = self._client._call_with_host_and_auth(method=task_api.get_job_info, job_name=self.job_name, job_id=self.job_id)

        def await_success(self, timeout=timedelta(minutes=30), poll_interval=timedelta(seconds=10)):
            self.await_finish(timeout=timeout, poll_interval=poll_interval)
            self.check_successful()

        def await_finish(self, timeout=timedelta(minutes=30), poll_interval=timedelta(seconds=10)):
            finish_time = datetime.now() + timeout
            self._update()
            while finish_time > datetime.now() and not self._is_job_in_terminate_state():
                time.sleep(poll_interval.total_seconds())
                self._update()

            if self._is_job_in_terminate_state():
                return self._job
            else:
                raise JenkinsClientError(
                    "Awaited to job {job_id} ({queue_item_id}) for {timeout} sec, still no result"
                        .format(job_id=self.job_id, queue_item_id=self.queue_item_id, timeout=timeout.total_seconds())
                )

        def check_successful(self):
            if not self._job or self._job["result"] != "SUCCESS":
                raise JenkinsFailedJobError("Job {job_id} is failed!".format(job_id=self.job_id))

        def _is_job_in_terminate_state(self):
            return self._job is not None and self._job.get("result") in self._TERMINAL_STATES

        @staticmethod
        def _extract_job_id_from_queue_item(queue_item):
            executable = queue_item.get("executable")
            if executable is None:
                return None
            return executable.get("number")

    def __init__(self, host, user, token):
        self.host = self._normalize_host(host)
        self.user = user
        self.token = token
        self._auth = self._make_auth()

    @staticmethod
    def _normalize_host(host_raw):
        if host_raw.endswith('/'):
            host_raw = host_raw[:-1]
        return host_raw

    def run_job(self, job_name, parameters=None):
        if not parameters:
            parameters = None
        queue_item_id = task_api.build_job(host=self.host, auth=self._auth, job_name=job_name, parameters=parameters)
        return self._JobHandler(client=self, job_name=job_name, queue_item_id=queue_item_id)

    def get_job_stdout(self, job_name, job_id):
        return self._call_with_host_and_auth(
            method=task_api.download_job_stdout,
            job_name=job_name,
            job_id=job_id
        )

    def _make_auth(self):
        return HTTPBasicAuth(self.user, self.token)

    def _call_with_host_and_auth(self, method, *args, **kwargs):
        return method(host=self.host, auth=self._auth, *args, **kwargs)
