import logging
import os
import json
import datetime

from sandbox import sdk2
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.projects.tank.load_resources.resources import YANDEX_TANKAPI_LXC_CONTAINER, YANDEX_TANK_LOGS
import sandbox.projects.common.build.parameters as build_params
from sandbox.projects.BuildDockerImageV6 import BuildDockerImageV6
from sandbox.projects.mail.CommonLib.lib.tank_api import TankApi
from sandbox.projects.mail.CommonLib.lib.resources import MAIL_SHOOTING_MONITORING_INFO, MAIL_SHOOTING_AGGREGATE_INFO,\
    MAIL_SHOOTING_YASM_GET_SCRIPT, QLOUD_ENVIRONMENT_CONFIG
from sandbox.projects.mail.CommonLib.lib.qloud import QloudApi
from sandbox.projects.mail.CommonLib.lib.docker import get_image_hash

QLOUD_URL = 'https://platform.yandex-team.ru'
TANKAPI_TANK_PORT = '8083'


class ShingerPrintShooting(sdk2.Task):

    class Context(sdk2.Context):
        child_tasks_ids = []
        build_docker_image_tag = ''
        lunapark_url = ''
        build_revision = None

        min_user_memory_usage_at_base_rps = None
        avg_user_memory_usage_at_base_rps = None
        median_user_memory_usage_at_base_rps = None
        max_user_memory_usage_at_base_rps = None
        stddev_user_memory_usage_at_base_rps = None
        mem_guarantee = None

        min_user_cpu_usage_at_base_rps = None
        avg_user_cpu_usage_at_base_rps = None
        median_user_cpu_usage_at_base_rps = None
        max_user_cpu_usage_at_base_rps = None
        stddev_user_cpu_usage_at_base_rps = None
        cpu_guarantee = None

        avg_system_cpu_usage_at_base_rps = None
        avg_disk_usage_at_base_rps = None
        avg_disk_write_fs_at_base_rps = None
        avg_disk_read_fs_at_base_rps = None

        percentile_98_at_base_rps = None
        percentile_95_at_base_rps = None

        good_http_codes_percent_at_base_rps = None

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 30*60
        owner = "MAIL"
        description = "Task for performing ShingerPrint load testing with Ya.Tank"

        container = sdk2.parameters.Container(
            'LXC Container with tankapi',
            resource_type=YANDEX_TANKAPI_LXC_CONTAINER,
            required=True
        )
        with sdk2.parameters.Group("Shooting params") as shooting_block:
            st_task = sdk2.parameters.String(
                'Task in startrack for shooting',
                default='MAILDEV-1007',
                required=True)

            operator = sdk2.parameters.String(
                'Shooting operator',
                default='robot-gerrit',
                required=True)

            job_name = sdk2.parameters.String(
                'Job name',
                default='Regular ShingerPrint performance test',
                required=True)

            shooting_type = sdk2.parameters.String(
                'Shooting type',
                default='line(2000,2200,10)',
                required=True)

            ammo_file = sdk2.parameters.Url(
                'Ammo file url',
                default='https://storage-int.mds.yandex.net/get-load-ammo/15349/1f72349e699045b2940106f73a3f90c0',
                required=True)

            monitoring_script = sdk2.parameters.Resource(
                'Yasm-get monitoring script',
                resource_type=MAIL_SHOOTING_YASM_GET_SCRIPT,
                required=False)

            base_service_rps = sdk2.parameters.Integer(
                'Which rps service must hold',
                default=-1,
                required=True)

            is_regular_performance_test = sdk2.parameters.Bool(
                'Is this shooting - regular performance test',
                default=True,
                required=True)

        with sdk2.parameters.Group("Qloud params") as qloud_block:
            delete_environment = sdk2.parameters.Bool(
                'Delete env after shooting end',
                default=False,
                required=True)

            qloud_env_conf = sdk2.parameters.Resource(
                'Platform env config',
                resource_type=QLOUD_ENVIRONMENT_CONFIG,
                required=True)

        with sdk2.parameters.Group("Docker") as docker_block:
            arcadia_url = sdk2.parameters.ArcadiaUrl("Svn url for arcadia checkout",
                                                     default_value=build_params.ArcadiaUrl.default_value)

            build_docker_image = sdk2.parameters.Bool(
                'Should build new docker image and push it to repo',
                default=True,
                required=True)

            topic = sdk2.parameters.String("Topic for this release (optional)")

    def on_create(self):
        self.Parameters.container = YANDEX_TANKAPI_LXC_CONTAINER.find(
            status=ctr.State.READY,
            owner='LOAD-ROBOT'
        ).order(-YANDEX_TANKAPI_LXC_CONTAINER.id).first().id
        logging.info('Container %s is taken', self.Parameters.container.id)

        self.Parameters.monitoring_script = MAIL_SHOOTING_YASM_GET_SCRIPT.find(
            status=ctr.State.READY,
        ).order(-MAIL_SHOOTING_YASM_GET_SCRIPT.id).first().id
        logging.info('Monitoring script %s is taken', self.Parameters.monitoring_script.id)

        logging.info('Qloud environment config %s is taken', self.Parameters.qloud_env_conf.id)

    def _saving_resources(self, tank_api):
        folder_resource_data = sdk2.ResourceData(
            YANDEX_TANK_LOGS(self, 'Yandex tank logs for shoot {}'.format(tank_api.job_number), self.tanklogs_path)
        )

        folder_resource_data.ready()

        monitoring_resource = sdk2.ResourceData(
            MAIL_SHOOTING_MONITORING_INFO(
                self,
                "Monitoring info",
                str(self.path('shooting.monitoring.info.json')),
                ttl=14))
        monitoring_resource.path.write_bytes(json.dumps(tank_api.monitoring_report_json))
        monitoring_resource.ready()

        aggregate_info_resource = sdk2.ResourceData(
            MAIL_SHOOTING_AGGREGATE_INFO(
                self,
                "Aggregate shooting info",
                str(self.path('shooting.aggregate.info.json')),
                ttl=14))
        aggregate_info_resource.path.write_bytes(json.dumps(tank_api.common_info_report_json))
        aggregate_info_resource.ready()

    def _get_qloud_environment_config(self):
        qloud_env_conf_data = sdk2.resource.ResourceData(self.Parameters.qloud_env_conf)
        conf = qloud_env_conf_data.path.read_bytes()
        self.qloud_env_conf = json.loads(conf)
        if self.Parameters.build_docker_image and self.Context.build_docker_image_tag:
            for num, component in enumerate(self.qloud_env_conf['components']):
                if 'target' in component['componentName'].lower():
                    logging.debug("Update qloud target image repository and hash")
                    repo, tag = self.Context.build_docker_image_tag.split(':')
                    if not repo.startswith('registry.yandex.net'):
                        repo = 'registry.yandex.net/' + repo
                    component['properties']['repository'] = repo + ':' + tag
                    component['properties']['hash'] = \
                        get_image_hash(repo, tag)
                    logging.debug("Target image for deploy: " + component['properties']['repository'] + " hash: " +
                                  component['properties']['hash'])

        logging.info("Qloud environment config load.")
        return json.dumps(self.qloud_env_conf)

    def _set_tank_and_target_component_names(self):
        component_names = [self.qloud_env_conf['components'][0]['componentName'],
                           self.qloud_env_conf['components'][1]['componentName']]
        if 'tank' in component_names[0].lower():
            self._tank_name = component_names[0]
            self._target_name = component_names[1]
        else:
            self._tank_name = component_names[1]
            self._target_name = component_names[0]

    def _make_registry_tag(self, topic):
        tag = 'mail/shinger_print/shinger_print' + ":"
        tag += 'r' + str(self.Context.build_revision) + '.'
        tag += datetime.datetime.now().strftime("%Y%m%d-%H%M")
        if topic:
            tag += "." + topic
        return tag

    def __build_docker_image(self):
        logging.info("Start build docker image")
        task_class = sdk2.Task['BUILD_DOCKER_IMAGE_V6']
        registry_tag = self._make_registry_tag(self.Parameters.topic)
        kwargs = {
            BuildDockerImageV6.ArcadiaUrl.name: self.__arcadia_url,
            BuildDockerImageV6.DockerPackageJsonParameter.name: 'mail/apphost/shinger_print/package/package.json',
            BuildDockerImageV6.RegistryTags.name: [registry_tag],
            BuildDockerImageV6.RegistryLogin.name: "robot-gerrit",
            BuildDockerImageV6.VaultItemOwner.name: "MAIL",
            BuildDockerImageV6.VaultItemName.name: "robot-gerrit-oauth-registry",
        }
        build_img_task = task_class(
            self,
            description="Subtask for build shinger print docker image",
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            **kwargs
        ).enqueue()
        self.Context.child_tasks_ids.append(build_img_task.id)
        logging.info("Start wait subtask id:{}".format(build_img_task.id))
        raise sdk2.WaitTask(self.Context.child_tasks_ids, (ctt.Status.Group.FINISH | ctt.Status.Group.BREAK),
                            wait_all=True)

    def __fill_context_metrics(self, metrics):
        for metric_name in metrics:
            if 'http' in metric_name:
                for item in metrics[metric_name]:
                    if item['http'] == 200:
                        self.Context.good_http_codes_percent_at_base_rps = item['percent']
            elif 'percentile' in metric_name:
                for item in metrics[metric_name]:
                    if item['percentile'] == '98':
                        self.Context.percentile_98_at_base_rps = item['ms']
                    elif item['percentile'] == '95':
                        self.Context.percentile_95_at_base_rps = item['ms']
            elif 'cpu_user' in metric_name:
                if 'max' in metrics[metric_name]:
                    self.Context.max_user_cpu_usage_at_base_rps = metrics[metric_name]['max']
                if 'min' in metrics[metric_name]:
                    self.Context.min_user_cpu_usage_at_base_rps = metrics[metric_name]['min']
                if 'median' in metrics[metric_name]:
                    self.Context.median_user_cpu_usage_at_base_rps = metrics[metric_name]['median']
                if 'avg' in metrics[metric_name]:
                    self.Context.avg_user_cpu_usage_at_base_rps = metrics[metric_name]['avg']
                if 'stddev' in metrics[metric_name]:
                    self.Context.stddev_user_cpu_usage_at_base_rps = metrics[metric_name]['stddev']
            elif 'mem_usage' in metric_name:
                if 'max' in metrics[metric_name]:
                    self.Context.max_user_memory_usage_at_base_rps = metrics[metric_name]['max']
                if 'min' in metrics[metric_name]:
                    self.Context.min_user_memory_usage_at_base_rps = metrics[metric_name]['min']
                if 'median' in metrics[metric_name]:
                    self.Context.median_user_memory_usage_at_base_rps = metrics[metric_name]['median']
                if 'avg' in metrics[metric_name]:
                    self.Context.avg_user_memory_usage_at_base_rps = metrics[metric_name]['avg']
                if 'stddev' in metrics[metric_name]:
                    self.Context.stddev_user_memory_usage_at_base_rps = metrics[metric_name]['stddev']
            elif 'mem_guarantee' in metric_name:
                self.Context.mem_guarantee = metrics[metric_name]['avg']
            elif 'cpu_guarantee' in metric_name:
                self.Context.cpu_guarantee = metrics[metric_name]['avg']
            elif 'disk_write-fs' in metric_name:
                if 'avg' in metrics[metric_name]:
                    self.Context.avg_disk_write_fs_at_base_rps = metrics[metric_name]['avg']
            elif 'disk_read-fs' in metric_name:
                if 'avg' in metrics[metric_name]:
                    self.Context.avg_disk_read_fs_at_base_rps = metrics[metric_name]['avg']

    def __get_tank_and_target_addresses_and_hosts(self, qloud_api, environment_id, tank_name, target_name):
        res = qloud_api.get_all_running_instances(environment_id + '.' + tank_name)
        assert res.ok, {'status_code': res.status_code, 'text': res.text}
        tank_instance_address = json.loads(res.text)[0]['instanceFqdn']
        tank_instance_host = json.loads(res.text)[0]['hostFqdn']
        res = qloud_api.get_all_running_instances(environment_id + '.' + target_name)
        assert res.ok, {'status_code': res.status_code, 'text': res.text}
        target_instance_address = json.loads(res.text)[0]['instanceFqdn']
        target_instance_host = json.loads(res.text)[0]['hostFqdn']
        logging.debug('tank host: {tank_host} target host: {target_host}'.format(
            tank_host=tank_instance_host,
            target_host=target_instance_host
        ))
        return tank_instance_address, target_instance_address, tank_instance_host, target_instance_host

    def on_prepare(self):
        parsed_url = Arcadia.parse_url(self.Parameters.arcadia_url)
        if not parsed_url.revision:
            self.Context.build_revision = Arcadia.get_revision(self.Parameters.arcadia_url)
            logging.info('Revision for build docker image if need: ' + str(self.Context.build_revision))
            self.__arcadia_url = self.Parameters.arcadia_url + '@' + str(self.Context.build_revision)
        else:
            self.Context.build_revision = parsed_url.revision
            logging.info('Revision for build docker image if need: ' + str(parsed_url.revision))
            self.__arcadia_url = self.Parameters.arcadia_url

    def on_execute(self):
        logging.info("Shinger print job shooting start...")

        if self.Parameters.build_docker_image:
            with self.memoize_stage.build_docker_image:
                try:
                    self.__build_docker_image()
                except Exception:
                    logging.error("Can't build docker image.")
                    raise
            logging.info("Finish waiting build docker image task")
            sub_task = self.find().first()
            if not sub_task:
                is_error = True
            else:
                is_error = sub_task.status != ctt.Status.SUCCESS

            if is_error:
                raise Exception("Subtask finished without success")
            self.Context.build_docker_image_tag = sub_task.Context.registry_tags[0]

        with self.memoize_stage.upload_qloud_env:
            qloud_auth_token = sdk2.Vault.data("robot-gerrit-oauth-platform")

            qloud_api = QloudApi(QLOUD_URL, qloud_auth_token, logging.getLogger('QloudApi'))

            logging.info('Start creating qloud environment')
            qloud_env_conf = self._get_qloud_environment_config()
            qloud_api.upload_environment(qloud_env_conf)

            self._set_tank_and_target_component_names()

            self.tank_address, self.target_address, self.tank_host, self.target_host = \
                self.__get_tank_and_target_addresses_and_hosts(
                    qloud_api, self.qloud_env_conf['objectId'], self._tank_name, self._target_name)

        with self.memoize_stage.shooting:
            monitoring_script_resource = sdk2.resource.ResourceData(self.Parameters.monitoring_script)

            tank_api = TankApi(
                target_address=self.target_address,
                target_host=self.target_host,
                ammo_file=self.Parameters.ammo_file,
                task=self.Parameters.st_task,
                logger=logging.getLogger('TankApi'),
                tank_address=self.tank_address,
                tank_port=TANKAPI_TANK_PORT,
                shooting_type=self.Parameters.shooting_type,
                shooting_component_name='shinger-print',
                jobno='jobno',
                monitoring_script_path=str(monitoring_script_resource.path),
                job_name=self.Parameters.job_name,
                operator=self.Parameters.operator,
                is_regular_performance_test=self.Parameters.is_regular_performance_test,
                base_service_rps=self.Parameters.base_service_rps)

            tank_api.create_tank_config()
            tank_api.write_tank_config('load.ini')
            tank_api.start_shooting()
            tank_api.fill_shooting_reports()
            logging.info("Shinger print shooting end...")

        with self.memoize_stage.save_resources_update_ctx:
            self.tanklogs_path = str(self.path('tank_logs'))
            logging.info("Tank logs saving in {}".format(self.tanklogs_path))
            if not os.path.exists(self.tanklogs_path):
                os.makedirs(self.tanklogs_path, 0o755)

            tank_api.post_process_logs(str(self.path()), self.tanklogs_path)

            self.Context.lunapark_url = tank_api.shooting_link

            self._saving_resources(tank_api)

            if self.Parameters.base_service_rps:
                metrics = tank_api.get_metrics_for_base_rps()
                logging.info("Metrics: {}".format(json.dumps(metrics)))
                self.__fill_context_metrics(metrics)

        if self.Parameters.delete_environment:
            qloud_api.delete_environment(self.qloud_env_conf['objectId'])
