# coding=utf-8
import logging
from collections import namedtuple

from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.common.errors import TaskFailure
from sandbox.common.errors import VaultError

from sandbox.projects.mail.CommonLib.lib.docker import get_image_hash
from sandbox.projects.mail.CommonLib.lib.qloud import QloudApi
from sandbox.projects.mail.CommonLib.lib.star_track import StarTrackApi


Platform = namedtuple('Platform', 'title, url')

PLATFORMS = [
    Platform('Stable (platform)', 'https://platform.yandex-team.ru'),
    Platform('Prestable (Qloud)', 'https://qloud.yandex-team.ru')
]


class UpdateQloudComponent(sdk2.Task):
    """Update docker image in Qloud component and optionally report result to StarTrack"""

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 30*60
        owner = "MAIL"
        description = "Deploy docker image into Qloud component"
        priority = ctt.Priority(ctt.Priority.Class.USER, ctt.Priority.Subclass.NORMAL)
        image_url = sdk2.parameters.String("Full docker image url with tag (registry.yandex.net/blah/blah:tag)")
        with sdk2.parameters.Group("Qloud parameters") as qloud_block:
            components = sdk2.parameters.List("List of components (project.application.environment.component)")
            qloud_token_vault_item = sdk2.parameters.String("Vault item containing Qloud OAuth token")
            deploy_comment = sdk2.parameters.String("Deploy comment (optional)", default=None)
            with sdk2.parameters.String("Select platform installation", multiline=True,
                                        default=PLATFORMS[0].title) as platform:
                for p in PLATFORMS:
                    platform.values[p.title] = p.title
        with sdk2.parameters.Group("StarTrack parameters (optional)") as star_track_block:
            issues = sdk2.parameters.List("Issues id")
            st_token_vault_item = sdk2.parameters.String("Vault item containing ST OAuth token",
                                                         default='robot-gerrit-oauth-startrek')

    def _selected_platform(self):
        return filter(lambda p: p.title == self.Parameters.platform, PLATFORMS)[0]

    def _get_vault_data(self, vault_item):
        try:
            return sdk2.Vault.data(vault_item)
        except VaultError as e:
            raise TaskFailure("Invalid vault item {}: {}".format(vault_item, str(e)))

    def _make_environment_link(self, component):
        return "{}/projects/{}".format(
            self._selected_platform().url,
            self._get_environment_id(component).replace('.', '/'))

    def _make_issue_message(self, is_success):
        if is_success:
            comment = "!!(green)**SUCCESS**!! Qloud deploy finished successfully\n"
        else:
            comment = "!!**ERROR**!! Qloud deploy failed\n"
        comment += "**Image:**\n%%{}%%\n".format(self.Parameters.image_url)

        components_list = '\n'.join(map(lambda c: "**(({} {}))**".format(self._make_environment_link(c), c),
                                        self.Parameters.components))
        comment += "**Components:**\n{}\n".format(components_list)

        if self.Parameters.issues:
            comment += "**Tasks:**\n"
            for issue in self.Parameters.issues:
                comment += "{}\n".format(issue)
        return comment

    def _post_message(self, is_success):
        issues = self.Parameters.issues
        if issues:
            token = self._get_vault_data(self.Parameters.st_token_vault_item)
            api = StarTrackApi(token=token)
            message = self._make_issue_message(is_success)
            errors = api.add_comments(issues=issues, text=message, task_id=self.id)
            logging.error("Errors while commenting star track: {}".format(str(errors)))

    def _get_environment_id(self, component):
        pos = component.rfind('.')
        return component[:pos]

    def _get_component_id(self, component):
        pos = component.rfind('.') + 1
        return component[pos:]

    def _get_environment_deployed_version(self, qloud_api, component):
        environment_id = self._get_environment_id(component)
        version = qloud_api.get_environment_deployed_version(environment_id)
        if not version:
            raise TaskFailure("Environment {} deployed version not found".format(environment_id))
        return version

    def on_execute(self):
        token = self._get_vault_data(self.Parameters.qloud_token_vault_item)
        image_url = self.Parameters.image_url
        image_hash = get_image_hash(*image_url.split(':'))
        logging.info("Image {} hash resolved: {}".format(image_url, image_hash))
        platform = self._selected_platform()
        logging.info("Using {} ({}) platform installation".format(platform.title, platform.url))

        api = QloudApi(url=platform.url, token=token,
                       logger=logging.getLogger('QloudApi'))

        component_info = map(lambda c: (self._get_environment_id(c),
                                        self._get_component_id(c),
                                        self._get_environment_deployed_version(api, c)),
                             self.Parameters.components)

        for (environment_id, component_id, version) in component_info:
            api.deploy_component_image(image=image_url, image_hash=image_hash,
                                       environment_id=environment_id,
                                       component=component_id,
                                       version=version,
                                       comment=self.Parameters.deploy_comment)

        for (environment_id, _, _) in component_info:
            api.wait_for_environment_deploy(environment_id)

    # noinspection PyUnusedLocal
    def on_success(self, prev_status):
        self._post_message(is_success=True)

    # noinspection PyUnusedLocal
    def on_failure(self, prev_status):
        self._post_message(is_success=False)

    # noinspection PyUnusedLocal
    def on_break(self, prev_status, status):
        self._post_message(is_success=False)
