from collections import defaultdict, namedtuple
import datetime
import logging
import os
import re
import shlex
import shutil

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.maps.common.ecstatic_bin import MapsEcstaticToolMixin
from sandbox.projects.maps.common.juggler_alerts import TaskJugglerReportWithParameters
from sandbox.sdk2.helpers import ProcessLog, subprocess as sp
import sandbox.projects.common.arcadia.sdk as arcadia_sdk
import sandbox.projects.common.constants as consts
import sandbox.common.types.task as ctt


BINARY_LOGGER_NAME = 'binary'
DEFAULT_ECSTATIC_DATASET_VERSION_FORMAT = '%Y.%m.%d.%H.%M.%S'
LOCAL_ECSTATIC_DATASET_DIR = "ecstatic_dataset"
LOCAL_INFO_FILE = 'info'
LOCAL_RELEASE_DIR = 'release'
LOCAL_RESOURCE_DIR = "resource"
VERSION_RE = re.compile(r'^[a-zA-Z0-9\.\+\-\~]*$')
UPLOAD_AND_FAIL_RET_CODE = 77

TVM_IDS = {
    'testing': 2018420,
    'datatesting': 2018420,
    'production': 2018422,
}
TVM_SEC_UUIDS = {
    'testing': 'sec-01e0q9xznk7j2qjd196w49drf3',
    'datatesting': 'sec-01e0q9xznk7j2qjd196w49drf3',
    'production': 'sec-01e0q9z2q3wk1fbhzx7cek4kvb',
}


class AnalyzerRegularYtProcessPackage(sdk2.Resource):
    revision = sdk2.Attributes.Integer("Revision", required=True)
    build_path = sdk2.Attributes.String("Path(from Arcadia root) which should be built", required=True)


class AnalyzerRegularYtBuildPackage(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 8
        disk_space = 200 * 1024  # Mb
        ram = 16 * 1024  # Mb

    class Parameters(sdk2.Task.Parameters):
        revision = sdk2.parameters.Integer('Revision', required=True)
        build_path = sdk2.parameters.String('Path(from Arcadia root) which should be built', required=True)
        vault_owner = sdk2.parameters.String('Sandbox Vault owner', required=True)
        yt_token_vault_name = sdk2.parameters.String('YT token Sandbox Vault name', required=False, default='YT_TOKEN')

    def on_execute(self):
        resource = AnalyzerRegularYtProcessPackage(
            self,
            "binary {} revision {}".format(self.Parameters.build_path, self.Parameters.revision),
            LOCAL_RESOURCE_DIR,
            build_path=self.Parameters.build_path,
            revision=self.Parameters.revision
        )

        resource_data = sdk2.ResourceData(resource)

        with arcadia_sdk.mount_arc_path(sdk2.svn.Arcadia.ARCADIA_TRUNK_URL + '@{}'.format(self.Parameters.revision)) as arcadia:
            arcadia_sdk.do_build(
                consts.YMAKE_BUILD_SYSTEM,
                arcadia,
                [self.Parameters.build_path],
                consts.RELEASE_BUILD_TYPE,
                clear_build=False,
                results_dir=str(resource_data.path),
                yt_store_params=arcadia_sdk.YtStoreParams(
                    store=True,
                    token=sdk2.Vault.data(self.Parameters.vault_owner, self.Parameters.yt_token_vault_name),
                    proxy="hahn",
                    dir="//home/devtools/cache",
                    put=False,
                    codec=None,
                    replace_result=False,
                    max_cache_size=None,
                    store_exclusive=False
                )
            )
        resource_data.ready()


class AnalyzerRegularYtProcess(TaskJugglerReportWithParameters, MapsEcstaticToolMixin):
    # Multislot: https://wiki.yandex-team.ru/sandbox/cookbook/#cores1multislot/cores4/cores8
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(TaskJugglerReportWithParameters.Parameters):
        kill_timeout = 7 * 24 * 3600

        with sdk2.parameters.RadioGroup("Environment") as environment:
            environment.values.development = environment.Value(value='development', default=True)
            environment.values.datatesting = environment.Value(value='datatesting')
            environment.values.testing = environment.Value(value='testing')
            environment.values.production = environment.Value(value='production')

        build_path = sdk2.parameters.String('Path(from Arcadia root) which should be built', required=True)
        binary_path = sdk2.parameters.String('Path(from Arcadia root) to binary file', required=True)
        binary_params = sdk2.parameters.String('Parameters to binary', default="")
        ecstatic_dataset_name = sdk2.parameters.String('Ecstatic dataset name', default="")
        ecstatic_dataset_version_from_directory_name = sdk2.parameters.Bool('Ecstatic datasests/versions from directory names', default_value=False)

        revision = sdk2.parameters.Integer('Revision', required=True)
        vault_owner = sdk2.parameters.String('Sandbox Vault owner', required=True)
        yql_token_vault_name = sdk2.parameters.String('YQL token Sandbox Vault name', required=False, default='YQL_TOKEN')
        yt_token_vault_name = sdk2.parameters.String('YT token Sandbox Vault name', required=False, default='YT_TOKEN')
        statbox_token_vault_name = sdk2.parameters.String('Statbox token Sandbox Vault name', required=False, default='STATBOX_TOKEN')
        yavault_token_vault_name = sdk2.parameters.String('Yandex Vault OAuth token Sandbox Vault name', required=False, default='YAVAULT_TOKEN')
        nirvana_token_vault_name = sdk2.parameters.String('Nirvana token Sandbox Vault name', required=False, default='NIRVANA_TOKEN')
        always_include_mmapped_files = sdk2.parameters.Bool('Include mmapped files to memory limit',  default_value=False)
        use_physical_pools = sdk2.parameters.Bool('Use physical pools', default_value=False)
        mmapped_in_tmpfs = sdk2.parameters.Bool('Allow order files for YT operations directly in tmpfs',  default_value=False)

    def on_prepare(self):
        os.environ["YENV_TYPE"] = self.Parameters.environment if self.Parameters.environment != 'datatesting' else 'testing'
        os.environ["ENVIRONMENT_NAME"] = self.Parameters.environment if self.Parameters.environment != 'datatesting' else 'testing'
        os.environ["ALZ_API_SVN_REVISION"] = str(self.Parameters.revision)
        os.environ["ALZ_API_SANDBOX_TASK_ID"] = str(self.id)
        os.environ["ALZ_API_ALWAYS_INCLUDE_MMAPPED_FILES"] = str(int(self.Parameters.always_include_mmapped_files))
        os.environ["ALZ_API_USE_PHYSICAL_POOLS"] = str(int(self.Parameters.use_physical_pools))
        os.environ["ALZ_API_MMAPPED_IN_TMPFS"] = str(int(self.Parameters.mmapped_in_tmpfs))
        tokens = {
            "ALZ_API_YQL_TOKEN": self.Parameters.yql_token_vault_name,
            "ALZ_API_YT_TOKEN": self.Parameters.yt_token_vault_name,
            "ALZ_API_STATBOX_TOKEN": self.Parameters.statbox_token_vault_name,
            "ALZ_API_YAVAULT_TOKEN": self.Parameters.yavault_token_vault_name,
            "ALZ_API_NIRVANA_TOKEN": self.Parameters.nirvana_token_vault_name,
        }

        for environ, vault_name in tokens.items():
            if vault_name:
                os.environ[environ] = sdk2.Vault.data(self.Parameters.vault_owner, vault_name)

        # Allows one to use resources from a subprocess
        # https://wiki.yandex-team.ru/sandbox/cookbook/#kakpoluchitdannyesinxronizirovatresursizpodprocessazadachi
        os.environ['SYNCHROPHAZOTRON_PATH'] = str(self.synchrophazotron)

        os.mkdir(LOCAL_ECSTATIC_DATASET_DIR)
        os.chmod(LOCAL_ECSTATIC_DATASET_DIR, 0o777)
        os.environ['ALZ_API_ECSTATIC_DATASET_DIR'] = os.path.join(os.getcwd(), LOCAL_ECSTATIC_DATASET_DIR)

        open(LOCAL_INFO_FILE, 'w').close()
        os.chmod(LOCAL_INFO_FILE, 0o777)
        os.environ['ALZ_API_INFO_FILE'] = os.path.join(os.getcwd(), LOCAL_INFO_FILE)

    def find_resource(self):
        return AnalyzerRegularYtProcessPackage.find(
            state="READY",
            attrs=dict(build_path=self.Parameters.build_path, revision=self.Parameters.revision)
        ).first()

    def prepare_resource_with_package(self):
        resource = self.find_resource()
        if resource:
            return str(sdk2.ResourceData(resource).path)

        with self.memoize_stage.build_package:
            child = AnalyzerRegularYtBuildPackage(
                self,
                description="Build {} revision {} for {}".format(self.Parameters.build_path, self.Parameters.revision, self.id),
                owner=self.Parameters.owner,
                priority=self.Parameters.priority,
                notifications=self.Parameters.notifications,
                revision=self.Parameters.revision,
                build_path=self.Parameters.build_path,
                vault_owner=self.Parameters.vault_owner,
                yt_token_vault_name=self.Parameters.yt_token_vault_name
            ).enqueue()

            raise sdk2.WaitTask([child.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True,
                                timeout=14400)

        raise Exception("package resource not found")

    def prepare_package(self):
        resource_dir = self.prepare_resource_with_package()
        shutil.rmtree(LOCAL_RELEASE_DIR, ignore_errors=True)
        shutil.copytree(resource_dir, LOCAL_RELEASE_DIR, symlinks=True)
        for root, _, _ in os.walk(LOCAL_RELEASE_DIR):
            os.chmod(root, 0o777)
        return LOCAL_RELEASE_DIR

    def upload_to_ecstatic_if_needed(self):
        """
        Upload data in `LOCAL_ECSTATIC_DATASET_DIR` to ecstatic with version based on current time
        """
        Dataset = namedtuple('Dataset', ['name', 'version', 'path'])

        ecstatic_dataset_name = str(self.Parameters.ecstatic_dataset_name)
        try:
            # env of ecstatic differ from task and need to be set to "stable" for cookie extraction
            def upload(dataset):
                env = self.Parameters.environment if self.Parameters.environment != 'production' else 'stable'
                dataset_name_with_version = dataset.name + '=' + dataset.version
                self.set_info("Uploading to ecstatic dataset={}, env={}".format(dataset_name_with_version, env))
                self.ecstatic(
                    env,
                    ['upload', dataset_name_with_version, dataset.path, '+stable'],
                    tvm_id=TVM_IDS.get(self.Parameters.environment),
                    tvm_secret_id=TVM_SEC_UUIDS.get(self.Parameters.environment),
                )

            def process_dataset(dirname):
                data_path = os.path.join(LOCAL_ECSTATIC_DATASET_DIR, dirname)
                assert os.path.isdir(data_path), \
                    'Local ecstatic content is not a directory: {}'.format(dirname)
                split = dirname.split('=')

                if len(split) == 1:  # only version specified
                    [version] = split
                    assert ecstatic_dataset_name, \
                        '`ecstatic_dataset_name` should be specified if only version provided in directory name: {}'.format(dirname)
                    dataset_name = ecstatic_dataset_name
                else:  # dataset with version specified
                    assert len(split) == 2, 'incorrect directory name: {}, `=` should appear not more than once'.format(dirname)
                    if ecstatic_dataset_name:
                        self.set_info('Dataset name "{}" will be ignored while processing "{}"'.format(ecstatic_dataset_name, dirname))
                    [dataset_name, version] = split

                assert VERSION_RE.match(version), 'incorrect version: {}'.format(version)
                return Dataset(dataset_name, version, data_path)

            if self.Parameters.ecstatic_dataset_version_from_directory_name:
                datasets = map(process_dataset, os.listdir(LOCAL_ECSTATIC_DATASET_DIR))

                def reducer(lhs, rhs):
                    lhs[rhs.name].append(rhs.version)
                    return lhs

                for dname, vs in reduce(reducer, datasets, defaultdict(list)).items():
                    assert len(vs) == 1, 'dataset "{}" version is ambiguous: {}'.format(dname, vs)

                for ds in datasets:
                    upload(ds)

            elif ecstatic_dataset_name:
                upload(Dataset(ecstatic_dataset_name, self.generate_version(), LOCAL_ECSTATIC_DATASET_DIR))

        except sp.CalledProcessError as e:
            self.set_info(e.output)
            raise TaskFailure('Ecstatic returned ' + str(e.returncode))

    def on_execute(self):
        package_dir = self.prepare_package()
        cmd = [self.Parameters.binary_path] + shlex.split(self.Parameters.binary_params)

        with ProcessLog(logger=logging.getLogger(BINARY_LOGGER_NAME)) as pl:
            ret_code = sp.call(
                cmd, env=os.environ, stdout=pl.stdout, stderr=pl.stdout, cwd=package_dir,
            )

        with open(LOCAL_INFO_FILE, 'r') as fin:
            info = fin.read()

        if info:
            self.set_info('Process returned info:\n{}'.format(info))

        if ret_code in [0, UPLOAD_AND_FAIL_RET_CODE]:
            self.upload_to_ecstatic_if_needed()

        if ret_code != 0:
            raise sp.CalledProcessError(cmd=cmd, returncode=ret_code)

    def generate_version(self):
        return datetime.datetime.utcnow().strftime(DEFAULT_ECSTATIC_DATASET_VERSION_FORMAT)


# Resource classes that are generated in the analyzer/sandbox tasks.

class MAPS_ANALYZER_BUILD_CALENDAR_RESOURCE(sdk2.Resource):
    """
        Calendar resource type
    """
    ttl = 'inf'
    environment = sdk2.parameters.String("environment", default="dev")


class MAPS_ANALYZER_BUILD_LIGHT_CALENDAR_RESOURCE(sdk2.Resource):
    """
        Light calendar resource type
    """
    ttl = 'inf'
    environment = sdk2.parameters.String("environment", default="dev")
