import logging
import os
import shlex

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog, subprocess as sp
import sandbox.projects.common.arcadia.sdk as arcadia_sdk
import sandbox.projects.common.constants as consts

BINARY_LOGGER_NAME = 'binary'
LOCAL_RELEASE_DIR = 'release'


class AnalyzerRegularYtProcessAapi(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 7 * 24 * 3600

        with sdk2.parameters.RadioGroup("Environment") as environment:
            environment.values.development = environment.Value(value='development', default=True)
            environment.values.testing = environment.Value(value='testing')
            environment.values.production = environment.Value(value='production')

        build_path = sdk2.parameters.String('Path(from Arcadia root) which should be built', required=True)
        binary_path = sdk2.parameters.String('Path(from Arcadia root) to binary file', required=True)
        binary_params = sdk2.parameters.String('Parameters to binary', default="")

        revision = sdk2.parameters.Integer('Revision', required=True)
        vault_owner = sdk2.parameters.String('Sandbox Vault owner', required=True)
        yql_token_vault_name = sdk2.parameters.String('YQL token Sandbox Vault name', required=True, default='YQL_TOKEN')
        yt_token_vault_name = sdk2.parameters.String('YT token Sandbox Vault name', required=True, default='YT_TOKEN')
        statbox_token_vault_name = sdk2.parameters.String('Statbox token Sandbox Vault name', required=True, default='STATBOX_TOKEN')

    def on_prepare(self):
        os.environ["YENV_TYPE"] = self.Parameters.environment
        os.environ["ALZ_API_YQL_TOKEN"] = sdk2.Vault.data(
            self.Parameters.vault_owner, self.Parameters.yql_token_vault_name
        )
        os.environ["ALZ_API_YT_TOKEN"] = sdk2.Vault.data(
            self.Parameters.vault_owner, self.Parameters.yt_token_vault_name
        )
        os.environ["ALZ_API_STATBOX_TOKEN"] = sdk2.Vault.data(
            self.Parameters.vault_owner, self.Parameters.statbox_token_vault_name
        )

    def on_execute(self):
        with arcadia_sdk.mount_arc_path(sdk2.svn.Arcadia.ARCADIA_TRUNK_URL + '@{}'.format(self.Parameters.revision)) as mount_path:
            arcadia_sdk.do_build(
                consts.DISTBUILD_BUILD_SYSTEM,
                mount_path,
                [self.Parameters.build_path],
                clear_build=False,
                results_dir=LOCAL_RELEASE_DIR
            )
            with ProcessLog(logger=logging.getLogger(BINARY_LOGGER_NAME)) as pl:
                sp.check_call(
                    [os.path.join(LOCAL_RELEASE_DIR, self.Parameters.binary_path)] + shlex.split(self.Parameters.binary_params),
                    env=os.environ, stdout=pl.stdout, stderr=pl.stdout
                )
