import os
import logging
import tarfile

from sandbox import sdk2
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.sdk2.helpers import subprocess
from sandbox.sdk2.helpers import ProcessLog
from sandbox.sandboxsdk.svn import Svn


class AnsibleJugglerRunner(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        aj_client = sdk2.parameters.Resource('Resource with ansible-juggler')
        pb_repo = sdk2.parameters.SvnUrl('Path to playbooks in arcadia')
        playbook = sdk2.parameters.String('Name of playbook to run')
        oauth_key = sdk2.parameters.String('Name of OAuth key in Vault')

    def on_execute(self):
        # place all data in task dir - it will be cleared by sandbox itself
        pb_root = str(self.path('playbooks'))
        playbook = os.path.join(pb_root, self.Parameters.playbook)
        logging.info(pb_root)

        Svn.checkout(url=self.Parameters.pb_repo, path=pb_root)
        logging.info('Got playbooks:')
        logging.info(os.listdir(pb_root))

        aj = sdk2.ResourceData(self.Parameters.aj_client)
        logging.info('Got resource - %s:%s' % (aj, str(aj.path)))
        aj_path = str(self.path('aj'))
        tarfile.open(str(aj.path)).extractall(aj_path)
        logging.info(os.listdir(aj_path))

        py_path = os.path.join(aj_path, 'bin', 'python')
        apb_path = os.path.join(aj_path, 'bin', 'ansible-playbook')
        logging.info(py_path, apb_path)

        apb_cmd = [py_path, apb_path, '-vvvvvv', '-i', '/dev/null', playbook]

        # get OAuth token for access to namespace in juggler
        os.environ['JUGGLER_OAUTH_TOKEN'] = sdk2.Vault.data(self.Parameters.oauth_key)

        # check all essential files in place
        logging.info(map(
            lambda x: os.path.exists(x) and x or open(x),  # can't raise exception in lamda, so, try open nonexisting file which will raise it for us
            [py_path, apb_path, playbook]))

        try:
            with ProcessLog(self, logger=logging.getLogger('ansible-playbook')) as pl:
                subprocess.check_call(apb_cmd, stdout=pl.stdout, stderr=pl.stderr)
        # if ansible-playbook died try to get log of last task it was applying
        except subprocess.CalledProcessError, ansible_err:
            ansible_log = []

            for line in reversed(fu.read_lines(str(self.log_path('common.log')))):
                if '(ansible-playbook)' in line:
                    ansible_log.insert(0, line)
                    if 'TASK: [' in line: break

            if ansible_log:
                eh.fail('\n'.join(ansible_log))
            else:
                raise ansible_err
