import os
import logging

from datetime import datetime

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog, subprocess
from sandbox.common.errors import ResourceNotFound, TaskError
from sandbox.projects.maps.common.juggler_alerts import (
    TaskJugglerReportWithParameters
)
from sandbox.projects.maps.common.latest_resources import find_latest_resource

class OrdersEntrancesMatcherBinary(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class FlatRangeBuilderBinary(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class FlatRangeStatisticsBinary(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


DATETIME_FORMAT = '%Y-%m-%d'


class BuildFlatsExport(TaskJugglerReportWithParameters):
    class Requirements(sdk2.Task.Requirements):
        ram = 8 << 10  # 8 GB
        cores = 8

    class Parameters(TaskJugglerReportWithParameters.Parameters):
        with sdk2.parameters.RadioGroup('Environment') as environment:
            environment.values["testing"] = environment.Value("testing", default=True)
            environment.values["stable"] = environment.Value("stable")

        with sdk2.parameters.Group('Binaries') as binaries:
            matcher_binary = sdk2.parameters.Resource(
                'maps/poi/flat_ranges/orders_entrances_matcher binary',
                resource_type=OrdersEntrancesMatcherBinary,
                required=False)

            flat_ranges_builder_binary = sdk2.parameters.Resource(
                'maps/poi/flat_ranges/builder binary',
                resource_type=FlatRangeBuilderBinary,
                required=False)

            flat_ranges_statistics_binary = sdk2.parameters.Resource(
                'maps/poi/flat_ranges/statistics binary',
                resource_type=FlatRangeStatisticsBinary,
                required=False)

        with sdk2.parameters.Group('Tokens') as tokens:
            vault_owner = sdk2.parameters.String(
                'Sandbox Vault owner',
                required=True,
                default='MAPS')
            yt_token = sdk2.parameters.Vault(
                'YT token vault name',
                required=True)
            yql_token = sdk2.parameters.Vault(
                'YQL token vault name',
                required=True)
            stat_token = sdk2.parameters.Vault(
                'Statface token vault name')

        with sdk2.parameters.Group('Input parameters') as input_parameters:
            input_tables = sdk2.parameters.List(
                'YT path to the Yandex.Eda logs',
                required=True)

        with sdk2.parameters.Group('Output parameters') as output_parameters:
            output_folder = sdk2.parameters.String(
                'YT path to the folder where to store the pipeline output',
                required=True)

            force_date = sdk2.parameters.String(
                'Output date ({}). If it is not set, when current date is used.'.format(DATETIME_FORMAT),
                required=False)

        with sdk2.parameters.Group('Matcher parameters') as matcher_parameters:
            matcher_cache_path = sdk2.parameters.String(
                'YT path where to store the Matcher cache',
                required=True)
            matcher_output_table = sdk2.parameters.String(
                'YT path where to store the Matcher output',
                required=True)
            matcher_unmatched_table = sdk2.parameters.String(
                'YT path where to store unmatched addresses',
                required=True)

        with sdk2.parameters.Group('Builder parameters') as builder_parameters:
            min_flat_range_gap_to_split = sdk2.parameters.Integer(
                'Mininmal gap length in flat range to split',
                default=2)

            min_flat_range_length = sdk2.parameters.Integer(
                'Minimal flat range length',
                default=1)

            preferred_countries = sdk2.parameters.List(
                'Preferred list of country isocodes',
                default=[])

            pool_size_limit = sdk2.parameters.Integer(
                'Limit of YT operations launched at the same time')

        with sdk2.parameters.Group('Statistics parameters') as statistics_parameters:
            output_folded_path = sdk2.parameters.String(
                'Path to save the folded table to on Hahn (if given)')

            stat_report_path = sdk2.parameters.String(
                'Path to publish the report to (if given)')

            with sdk2.parameters.RadioGroup('Statface proxy to use') as stat_proxy:
                stat_proxy.values.prod = stat_proxy.Value(
                    value='prod')
                stat_proxy.values.beta = stat_proxy.Value(
                    value='beta', default=True)

    def get_vault_secret(self, vault_name):
        return sdk2.Vault.data(self.Parameters.vault_owner, vault_name)

    def ensure_latest_resources_used(self):
        self.matcher_binary = self.Parameters.matcher_binary
        if not self.matcher_binary:
            self.matcher_binary = find_latest_resource(
                OrdersEntrancesMatcherBinary, self.Parameters.environment)
        self.flat_ranges_builder_binary = self.Parameters.flat_ranges_builder_binary
        if not self.flat_ranges_builder_binary:
            self.flat_ranges_builder_binary = find_latest_resource(
                FlatRangeBuilderBinary, self.Parameters.environment)
        self.flat_ranges_statistics_binary = self.Parameters.flat_ranges_statistics_binary
        if not self.flat_ranges_statistics_binary:
            self.flat_ranges_statistics_binary = find_latest_resource(
                FlatRangeStatisticsBinary, self.Parameters.environment)
        logging.info(
            'Working in %s environment', self.Parameters.environment)
        logging.info(
            'Using OrdersEntrancesMatcherBinary: %s',
            self.matcher_binary.id)
        logging.info(
            'Using FlatRangeBuilderBinary: %s',
            self.flat_ranges_builder_binary.id)
        logging.info(
            'Using FlatRangeStatisticsBinary: %s',
            self.flat_ranges_statistics_binary.id)

    def run_executable(self, executable_resource, executable_arguments, env_variables):
        if not executable_resource:
            raise ResourceNotFound('Executable resource was not found. Please provide a valid executable.')

        # ensure every argument is str
        executable_arguments = [str(arg) for arg in executable_arguments]

        executable = str(sdk2.ResourceData(executable_resource).path)
        args = [executable] + executable_arguments

        os.environ.update(env_variables)

        with ProcessLog(self, 'Executable') as pl:
            return_code = subprocess.call(
                args, env=os.environ, stdout=pl.stdout, stderr=pl.stderr)

        if return_code != 0:
            raise TaskError(
                'Executable {} failed with {} return code'.format(
                    args, return_code))

    def on_execute(self):
        output_date = datetime.utcnow().strftime(DATETIME_FORMAT)
        if self.Parameters.force_date:
            output_date = self.Parameters.force_date

        default_tokens = {
            'YT_TOKEN': self.get_vault_secret(self.Parameters.yt_token)
        }

        self.ensure_latest_resources_used()

        # running matcher binary first
        matcher_arguments = ['--input-tables']
        matcher_arguments.extend(self.Parameters.input_tables)
        matcher_arguments += [
            '--addresses-cache', self.Parameters.matcher_cache_path,
            '--output-table', self.Parameters.matcher_output_table,
            '--unmatched-table', self.Parameters.matcher_unmatched_table,
        ]

        matcher_tokens = {
            'YQL_TOKEN': self.get_vault_secret(self.Parameters.yql_token)
        }
        matcher_tokens.update(default_tokens)
        self.run_executable(
            executable_resource=self.matcher_binary,
            executable_arguments=matcher_arguments,
            env_variables=matcher_tokens)

        # running builder binary
        builder_arguments = [
            '--matched-orders-table', self.Parameters.matcher_output_table,
            '--output-dir', os.path.join(self.Parameters.output_folder, output_date),
            '--min-flat-range-gap-to-split', self.Parameters.min_flat_range_gap_to_split,
            '--min-flat-range-length', self.Parameters.min_flat_range_length,
        ]
        if self.Parameters.preferred_countries:
            builder_arguments += ['--preferred-countries']
            builder_arguments.extend(self.Parameters.preferred_countries)
        if self.Parameters.pool_size_limit:
            builder_arguments += [
                '--pool-size-limit', self.Parameters.pool_size_limit
            ]

        self.run_executable(
            executable_resource=self.flat_ranges_builder_binary,
            executable_arguments=builder_arguments,
            env_variables=default_tokens)

        if self.flat_ranges_statistics_binary:
            # running statistics binary
            statistics_arguments = [
                '--flat-ranges-export-data-path', self.Parameters.output_folder,
                '--statface-proxy', self.Parameters.stat_proxy,
            ]
            statistics_tokens = {}
            statistics_tokens.update(default_tokens)

            if self.Parameters.output_folded_path:
                statistics_arguments += ['--output-folded-path', self.Parameters.output_folded_path]
            if self.Parameters.stat_report_path:
                statistics_arguments += ['--statface-report-path', self.Parameters.stat_report_path]
                assert self.Parameters.stat_token, 'Please provide Statface token'
                statistics_tokens['STAT_TOKEN'] = self.get_vault_secret(self.Parameters.stat_token)

            self.run_executable(
                executable_resource=self.flat_ranges_statistics_binary,
                executable_arguments=statistics_arguments,
                env_variables=statistics_tokens)
