import itertools
import logging
import datetime
import os
import re
import time

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog, subprocess

from sandbox.projects.maps.common.jams_common import JamsBuildTaskCommon


UNARCHIVED_DIR = 'unarchived'

WEEK_IN_SECONDS = 7 * 24 * 60 * 60

DATASET_NAME = 'yandex-maps-jams-historic-maps'

TVM_IDS = {
    'testing': 2020723,
    'stable': 2020725,
}
TVM_SEC_UUIDS = {
    'testing': 'sec-01ea26gzdf2jyd66y1n8dxr2g8',
    'stable': 'sec-01ea26hncy9ejyx4c0h4a0m44r',
}


def to_timestamp(time_string):
    try:
        return int(time_string)
    except ValueError:
        return int(time.mktime(
            datetime.datetime.strptime(time_string, '%Y-%m-%d').timetuple()))


def to_day_and_version(s):
    result = re.match('yandex-maps-jams-historic-maps:([0-6])=([0-9-]+)', s)
    if result is None:
        return None

    return int(result.group(1)), result.group(2)


class BuildHistoricMaps(JamsBuildTaskCommon):

    class Parameters(JamsBuildTaskCommon.Parameters):
        force_generate_days = sdk2.parameters.List('Force generate data for these days', default=[])

    def on_execute(self):
        self.binary_resource_path = str(os.path.abspath(self.binary_resource()))

        self.prepare_ecstatic(TVM_IDS, TVM_SEC_UUIDS)

        today = datetime.datetime.now().strftime('%Y-%m-%d')
        days = self.days_for_data_generation(today)
        self.set_info('Days: {}'.format(','.join(days)))

        self.create_historic_maps(days)

        self.upload_maps()

    def create_historic_maps(self, days):
        create = self.binary_path('maps/jams/renderer2/historic/create_map/bin/create_historic_maps')

        geoid = self.binary_path('trf/geoid.mms.1')
        trf = self.binary_path('trf/trf.mms.1')
        geodata = self.binary_path('geodata/geodata6.bin')
        unarchived = str(os.path.abspath(UNARCHIVED_DIR))

        with ProcessLog(logger=logging.getLogger('create_historic_maps')) as log:
            cmd = [
                create,
                '--environment', self.Parameters.environment,
                '--geodata', geodata,
                '--geoid', geoid,
                '--unarchived', unarchived,
                '--trf', trf] + list(itertools.chain.from_iterable([('--days', d) for d in days]))
            subprocess.check_call(cmd, stdout=log.stdout, stderr=log.stderr)

    def upload_maps(self):
        version = int(time.time())
        for day in os.listdir(UNARCHIVED_DIR):
            path = os.path.join(UNARCHIVED_DIR, day)
            if os.path.isdir(path) and day in [str(i) for i in range(7)]:
                self.ecstatic_upload(path, DATASET_NAME, version, day)

    def binary_path(self, path):
        return os.path.join(self.binary_resource_path, self.Parameters.binary_package_path, path)

    def days_for_data_generation(self, today):
        days_and_version = [to_day_and_version(s) for s in self.ecstatic_versions(DATASET_NAME)
                            if to_day_and_version(s) is not None]
        today_timestamp = to_timestamp(today)

        def need_to_generate(day):
            if str(day) in self.Parameters.force_generate_days:
                return True

            if datetime.datetime.today().weekday() == day:
                return True

            versions_for_day = [to_timestamp(version) for d, version in days_and_version if d == day]
            if not versions_for_day:
                return True

            max_version = max(versions_for_day)
            return today_timestamp - max_version > WEEK_IN_SECONDS

        return [str(day) for day in xrange(7)
                if need_to_generate(day)]
