import datetime
import logging
import os
import shutil
import subprocess
import stat
import time

from sandbox import sdk2
from sandbox.common.types.task import Status
from sandbox.common.errors import TaskError

from sandbox.projects.maps.common.latest_resources import find_latest_resource
from sandbox.projects.maps.common.juggler_alerts import (
    TaskJugglerReportWithParameters
)


# This is not relevant in most cases, but there are quite intense YT-tasks
BUILD_STATIC_POI_ALL_TASKS_TIMEOUT_SECONDS = 20 * 60 * 60  # 20 hours


class BuildStaticPoiExecutable(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class ExperimentsHandlerExecutable(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class BuildStaticPoiTask(sdk2.Task):
    # requrements for MULTISLOT tag
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024  # MB

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        executable = sdk2.parameters.Resource(
            'Sandbox resource ID of the executable',
            resource_type=BuildStaticPoiExecutable,
            required=True
        )

        pool = sdk2.parameters.String(
            'YT pool to utilize', default='maps-core-static-poi', required=True
        )
        yt_vault = sdk2.parameters.Vault('YT vault', required=True)
        yql_vault = sdk2.parameters.Vault('YQL vault', required=True)

        organization_user_features_table = sdk2.parameters.String(
            'Organization-user features table', required=True
        )
        organizations_info_table = sdk2.parameters.String(
            'Organizations info table', required=True
        )
        iso_codes = sdk2.parameters.String(
            'ISO country codes', default='RU,UA,BY,TR,GH'
        )
        iso_codes_with_simplified_filtering = sdk2.parameters.String(
            'ISO country codes with simplified filtering', default='TR,GH'
        )
        ymapsdf_folder = sdk2.parameters.String(
            'Ymapsdf folder in YT', default='//home/maps/core/garden/stable/ymapsdf/latest'
        )
        yt_temp_folder_path = sdk2.parameters.String(
            'Path to yt folder where temp files will be stored'
        )

        now_utc_time = sdk2.parameters.String('Current time in UTC format')
        experiment = sdk2.parameters.String('Experiment name')
        update_info = sdk2.parameters.Bool(
            'Do you want to apply the experiment to the main data?',
            default=False
        )

        output_table = sdk2.parameters.String('Output table', required=True)
        history_dir = sdk2.parameters.String('History directory')
        extra_poi_bundle_folder = sdk2.parameters.String('Extra POI bundle folder')
        extra_poi_nyak_export_folder = sdk2.parameters.String('Extra POI nyak export folder')

    def on_execute(self):
        resource = sdk2.Resource[self.Parameters.executable]
        resource_data = sdk2.ResourceData(resource)
        executable_path = './executable'
        shutil.copyfile(str(resource_data.path), executable_path)
        os.chmod(
            executable_path, os.stat(executable_path).st_mode | stat.S_IEXEC
        )

        cmd = [
            executable_path,
            '--organization-user-features-table',
            self.Parameters.organization_user_features_table,
            '--organizations-info-table',
            self.Parameters.organizations_info_table,
            '--iso-codes',
            self.Parameters.iso_codes,
            '--iso-codes-with-simplified-filtering',
            self.Parameters.iso_codes_with_simplified_filtering,
            '--ymapsdf-folder',
            self.Parameters.ymapsdf_folder,
            '--output-table',
            self.Parameters.output_table
        ]

        if self.Parameters.now_utc_time:
            cmd += ['--now-utc-time', self.Parameters.now_utc_time]
        if self.Parameters.experiment:
            cmd += ['--experiment', self.Parameters.experiment]
        if self.Parameters.update_info:
            cmd.append('--update-info')
        if self.Parameters.pool:
            cmd += ['--pool', self.Parameters.pool]

        if self.Parameters.history_dir:
            cmd += ['--history-dir', self.Parameters.history_dir]
        if self.Parameters.extra_poi_bundle_folder:
            cmd += ['--extra-poi-bundle-folder', self.Parameters.extra_poi_bundle_folder]
        if self.Parameters.extra_poi_nyak_export_folder:
            cmd += ['--extra-poi-nyak-export-folder', self.Parameters.extra_poi_nyak_export_folder]
        if self.Parameters.yt_temp_folder_path:
            cmd += ['--yt-temp-path', self.Parameters.yt_temp_folder_path]

        os.environ['YT_TOKEN'] = self.Parameters.yt_vault.data()
        os.environ['YQL_TOKEN'] = self.Parameters.yql_vault.data()

        with sdk2.helpers.ProcessLog(
            self, logger=logging.getLogger('executable')
        ) as pl:
            subprocess.check_call(
                cmd, stdout=pl.stdout, stderr=subprocess.STDOUT
            )


class BuildStaticPoi(TaskJugglerReportWithParameters):
    class Context(sdk2.Context):
        now_utc_time = None

    class Parameters(TaskJugglerReportWithParameters.Parameters):
        with sdk2.parameters.RadioGroup('Environment') as environment:
            environment.values["testing"] = environment.Value("testing", default=True)
            environment.values["stable"] = environment.Value("stable")

        with sdk2.parameters.Group('Executable resources') as executable_resources:
            build_static_poi_executable = sdk2.parameters.Resource(
                'Sandbox resource ID of build_static_poi executable',
                resource_type=BuildStaticPoiExecutable,
                required=False
            )
            experiments_handler_executable = sdk2.parameters.Resource(
                'Sandbox resource ID of experiments_handler executable',
                resource_type=ExperimentsHandlerExecutable,
                required=False
            )

        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            pool = sdk2.parameters.String(
                'YT pool to utilize', default='maps-core-static-poi', required=True
            )
            yt_vault = sdk2.parameters.Vault('YT vault', required=True)
            yql_vault = sdk2.parameters.Vault('YQL vault', required=True)

        with sdk2.parameters.Group('Input parameters') as input_parameters:
            organization_user_features_table = sdk2.parameters.String(
                'Organization-user features table', required=True
            )
            organizations_info_table = sdk2.parameters.String(
                'Organizations info table', required=True
            )
            iso_codes = sdk2.parameters.String(
                'ISO country codes', default='RU,UA,BY,TR,GH'
            )
            iso_codes_with_simplified_filtering = sdk2.parameters.String(
                'ISO country codes with simplified filtering', default='TR,GH'
            )
            ymapsdf_folder = sdk2.parameters.String(
                'Ymapsdf folder in YT', default='//home/maps/core/garden/stable/ymapsdf/latest'
            )
            yt_temp_folder_path = sdk2.parameters.String(
                'Path to yt folder where temp files will be stored'
            )

        with sdk2.parameters.Group('Experiments parameters') as experiment_parameters:
            experiments = sdk2.parameters.List(
                'Experiments for POI ranking', sdk2.parameters.String, default=[]
            )
            apply_experiment = sdk2.parameters.String('Apply experiment to the main data')

        with sdk2.parameters.Group('Output parameters') as output_parameters:
            output_table = sdk2.parameters.String('Output table', required=True)
            history_dir = sdk2.parameters.String('History directory')
            extra_poi_bundle_folder = sdk2.parameters.String('Extra POI bundle folder')
            extra_poi_bundle_tmp_folder = sdk2.parameters.String('Extra POI bundle tmp folder')
            extra_poi_nyak_export_folder = sdk2.parameters.String('Extra POI nyak export folder')

    def check_subtasks(self):
        if not all(
            task.status == Status.SUCCESS
            for task in self.find(parent_id=self.id)
        ):
            raise TaskError('Child task failed')

    def ensure_latest_resources_used(self):
        self.build_static_poi_executable = self.Parameters.build_static_poi_executable
        if not self.build_static_poi_executable:
            self.build_static_poi_executable = find_latest_resource(
                BuildStaticPoiExecutable, self.Parameters.environment)

        self.experiments_handler_executable = self.Parameters.experiments_handler_executable
        if not self.experiments_handler_executable:
            self.experiments_handler_executable = find_latest_resource(
                ExperimentsHandlerExecutable, self.Parameters.environment)

        logging.info(
            'Working in %s environment', self.Parameters.environment)
        logging.info(
            'Using BuildStaticPoiExecutable: %s',
            self.build_static_poi_executable.id)
        logging.info(
            'Using ExperimentsHandlerExecutable: %s',
            self.experiments_handler_executable.id)

    def on_execute(self):
        if self.Context.now_utc_time is None:
            timestamp = time.time()
            self.Context.now_utc_time = \
                datetime.datetime.utcfromtimestamp(timestamp).strftime('%Y-%m-%dT%H:%M:%S.%fZ')

        self.ensure_latest_resources_used()

        with self.memoize_stage.build_static_poi_with_experiments:
            logging.info('Start building static poi')
            experiments = [experiment for experiment in self.Parameters.experiments]
            if not self.Parameters.apply_experiment:
                experiments.append(None)

            build_static_poi_task_common_params = {
                'executable': self.build_static_poi_executable.id,
                'pool': self.Parameters.pool,
                'yt_vault': self.Parameters.yt_vault,
                'yql_vault': self.Parameters.yql_vault,
                'organization_user_features_table': self.Parameters.organization_user_features_table,
                'organizations_info_table': self.Parameters.organizations_info_table,
                'iso_codes': self.Parameters.iso_codes,
                'iso_codes_with_simplified_filtering': self.Parameters.iso_codes_with_simplified_filtering,
                'ymapsdf_folder': self.Parameters.ymapsdf_folder,
                'output_table': self.Parameters.output_table,
                'history_dir': self.Parameters.history_dir,
                'extra_poi_bundle_folder': self.Parameters.extra_poi_bundle_tmp_folder,
                'kill_timeout': self.Parameters.kill_timeout,
                'yt_temp_folder_path': self.Parameters.yt_temp_folder_path,
            }
            build_static_poi_task_common_params['now_utc_time'] = self.Context.now_utc_time

            static_poi_tasks_ids = []
            for experiment in experiments:
                logging.info("Build static poi, experiment %s", experiment)
                task_params = dict(build_static_poi_task_common_params)
                if experiment:
                    task_params['experiment'] = experiment
                    if self.Parameters.apply_experiment == experiment:
                        task_params['update_info'] = True
                else:
                    task_params['extra_poi_nyak_export_folder'] = self.Parameters.extra_poi_nyak_export_folder
                task = BuildStaticPoiTask(
                    self,
                    description='Build static poi, experiment {}'.format(experiment),
                    **task_params
                )
                task.enqueue()
                static_poi_tasks_ids.append(task.id)

            raise sdk2.WaitTask(
                static_poi_tasks_ids,
                Status.Group.FINISH + Status.Group.BREAK,
                wait_all=True,
                timeout=BUILD_STATIC_POI_ALL_TASKS_TIMEOUT_SECONDS
            )

        self.check_subtasks()

        with self.memoize_stage.process_experiments:
            resource = sdk2.Resource[self.experiments_handler_executable.id]
            resource_data = sdk2.ResourceData(resource)
            executable_path = './experiments_handler'
            shutil.copyfile(str(resource_data.path), executable_path)
            os.chmod(
                executable_path, os.stat(executable_path).st_mode | stat.S_IEXEC
            )
            os.environ['YT_TOKEN'] = self.Parameters.yt_vault.data()
            os.environ['YQL_TOKEN'] = self.Parameters.yql_vault.data()

            cmd = [executable_path]

            if self.Parameters.pool:
                cmd += ['--pool', self.Parameters.pool]

            cmd += [
                'merge', '--extra-poi-bundle-folder',
                os.path.join(
                    self.Parameters.extra_poi_bundle_tmp_folder,
                    self.Context.now_utc_time
                )
            ]

            with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('experiments_handler')
            ) as pl:
                subprocess.check_call(
                    cmd, stdout=pl.stdout, stderr=subprocess.STDOUT
                )

            cmd = [executable_path]
            if self.Parameters.pool:
                cmd += ['--pool', self.Parameters.pool]
            cmd += [
                'move', '--input-dir',
                os.path.join(
                    self.Parameters.extra_poi_bundle_tmp_folder,
                    self.Context.now_utc_time
                ),
                '--output-dir',
                os.path.join(
                    self.Parameters.extra_poi_bundle_folder,
                    self.Context.now_utc_time
                ),
                '--static-poi-link', self.Parameters.output_table,
                '--static-poi-history-dir', self.Parameters.history_dir
            ]

            with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('experiments_handler')
            ) as pl:
                subprocess.check_call(
                    cmd, stdout=pl.stdout, stderr=subprocess.STDOUT
                )
