import datetime
import logging
import os
import shutil
import stat
import subprocess

from sandbox import sdk2
from sandbox.projects.common import apihelpers
from sandbox.projects.maps.common.juggler_alerts import (
    TaskJugglerReportWithParameters
)
from sandbox.projects.maps.common.latest_resources import find_latest_resource


class BuildYellowZonesExecutable(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']
    release_subscribers = ['tswr']


class BuildYellowZones(TaskJugglerReportWithParameters):
    class Parameters(TaskJugglerReportWithParameters.Parameters):
        with sdk2.parameters.RadioGroup('Environment') as environment:
            environment.values["testing"] = environment.Value("testing", default=True)
            environment.values["stable"] = environment.Value("stable")

        executable = sdk2.parameters.Resource(
            'Sandbox resource ID of the executable',
            resource_type=BuildYellowZonesExecutable,
            required=False
        )

        yt_vault = sdk2.parameters.Vault('YT vault', required=True)
        yql_vault = sdk2.parameters.Vault('YQL vault', required=True)
        export_directory = sdk2.parameters.String('Export YT directory', required=True)
        export_directory_tmp = sdk2.parameters.String('Export tmp YT directory', required=True)
        kill_timeout = 4 * 60 * 60

    def ensure_latest_resources_used(self):
        self.executable = self.Parameters.executable
        if not self.executable:
            self.executable = find_latest_resource(
                BuildYellowZonesExecutable, self.Parameters.environment)

        logging.info(
            'Working in %s environment', self.Parameters.environment)
        logging.info(
            'Using BuildYellowZonesExecutable: %s',
            self.executable.id)

    def on_execute(self):
        self.ensure_latest_resources_used()
        resource = sdk2.Resource[self.executable.id]
        resource_data = sdk2.ResourceData(resource)
        executable_path = './executable'
        shutil.copyfile(str(resource_data.path), executable_path)
        os.chmod(
            executable_path, os.stat(executable_path).st_mode | stat.S_IEXEC)
        table_name = datetime.date.today().strftime('%Y-%m-%d')
        cmd = [
            executable_path,
            '--export_directory_tmp',
            os.path.join(self.Parameters.export_directory_tmp, table_name),
            '--export_directory',
            os.path.join(self.Parameters.export_directory, table_name),
            '--latest_path',
            os.path.join(self.Parameters.export_directory, 'latest'),
        ]

        os.environ['YT_TOKEN'] = self.Parameters.yt_vault.data()
        os.environ['YQL_TOKEN'] = self.Parameters.yql_vault.data()

        with sdk2.helpers.ProcessLog(
            self, logger=logging.getLogger('executable')
        ) as pl:
            subprocess.check_call(
                cmd, stdout=pl.stdout, stderr=subprocess.STDOUT)
