import os
import yaml
import tarfile
import requests

import sandbox.common.types.task as ctt

from sandbox import common
from sandbox import sdk2
from sandbox.common.types.misc import DnsType
from sandbox.projects.BuildDockerImageV6_01 import BuildDockerImageV6_01
from sandbox.projects.common.nanny import nanny


SEDEM_TOKEN_OWNER = "MAPS"
SEDEM_TOKEN_NAME = "SEDEM_SECRETS"
REGISTRY_API = 'https://registry.yandex.net/v2/'
EMPTY_BLOB_HASH = 'sha256:a3ed95caeb02ffe68cdd9fd84406680ae93d633cb16422d00e8a7c22955b46d4'


class BuildDockerImageAndPortoLayer(nanny.ReleaseToNannyTask, sdk2.Task):
    """
    Builds docker image and generate portoLayer
    """

    class Requirements(sdk2.Task.Requirements):
        dns = DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):

        svn_version = sdk2.parameters.Integer('SVN Revision')
        package_json = sdk2.parameters.String('Package paths, related to Arcadia')
        docker_name = sdk2.parameters.String("Docker name to publish (registry.yandex.net/<this name>:<svn revision>)")
        porto_layer_name = sdk2.parameters.String('Name of Porto Layer')

    def docker_name(self):
        return "{}:{}".format(self.Parameters.docker_name, self.Parameters.svn_version)

    def build_image(self):
        task_class = sdk2.Task['BUILD_DOCKER_IMAGE_V6_01']
        kwargs = {
            BuildDockerImageV6_01.RegistryTags.name: [self.docker_name()],
            BuildDockerImageV6_01.ArcadiaUrl.name: "arcadia:/arc/trunk/arcadia@{}".format(self.Parameters.svn_version),
            BuildDockerImageV6_01.DockerPackageJsonParameter.name: self.Parameters.package_json,
            BuildDockerImageV6_01.RegistryLogin.name: 'robot-maps-sandbox',
            BuildDockerImageV6_01.VaultItemName.name: 'robot-maps-sandbox-docker-oauth',
            BuildDockerImageV6_01.VaultItemOwner.name: 'MAPS'
        }

        # logging.info(str(kwargs))
        sub_task = task_class(
            self,
            description="Building and publishing Docker image from Arcadia",
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            **kwargs
        ).enqueue()
        self.Context.sub_task_id = sub_task.id
        raise sdk2.WaitTask([sub_task], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)


    def docker_registry_get(self, *args, **kwargs):
        url = REGISTRY_API + self.Parameters.docker_name + '/' + '/'.join([str(x) for x in args])
        return requests.get(url, headers={'Authorization': 'OAuth ' + self.auth}, **kwargs)


    def download_docker_layers(self, output_stream):
        manifest = self.docker_registry_get('manifests', self.Parameters.svn_version).json()
        layers = manifest['fsLayers']
        self.set_info('Download {} Docker Image Layers'.format(len(layers)))
        loaded_files = set()
        opaque_dirs = tuple()

        output_layer = tarfile.open(mode='w:gz', fileobj=output_stream)

        WH_NO = 0
        WH_FILE = 1
        WH_DIR = 2

        for index, layer in enumerate(layers):
            blob_hash = layer['blobSum']
            if blob_hash != EMPTY_BLOB_HASH:
                stream = self.docker_registry_get('blobs', blob_hash, stream=True)
                input_layer = tarfile.open(mode='r|*', fileobj=stream.raw)
                for tarinfo in input_layer:

                    name = tarinfo.name
                    if name.startswith('./'):
                        name = name[2:]
                    name = name.strip('/')

                    basename = os.path.basename(name)
                    wh_mark = WH_NO
                    if basename.startswith('.wh.'):
                        if basename == '.wh..wh..opq':
                            wh_mark = WH_DIR
                            name = os.path.dirname(name)
                        else:
                            wh_mark = WH_FILE
                            name = os.path.dirname(name) +'/' + basename[4:]

                    if name in loaded_files:
                        continue
                    if (name+'/').startswith(opaque_dirs):
                        continue

                    if wh_mark == WH_DIR:
                        opaque_dirs = tuple(list(opaque_dirs) + [name])
                    else:
                        loaded_files.add(name)

                    if wh_mark == WH_NO:
                        output_layer.addfile(tarinfo, input_layer.extractfile(tarinfo) if tarinfo.isfile() else None)

        output_layer.close()

    def publish_porto_layer(self):
        tar_name = self.Parameters.porto_layer_name+'.tar.gz'
        resource = sdk2.Resource["PORTO_LAYER"](self, 'Porto Layer for Docker Image', tar_name)

        # setattr(resource, 'name', self.Parameters.porto_layer_name)
        # setattr(resource, 'stack', self.Parameters.porto_layer_name)
        # setattr(resource, 'docker', self.self.docker_name())

        resource_data = sdk2.ResourceData(resource)
        with resource_data.path.open('wb') as f:
            self.download_docker_layers(f)
        resource_data.ready()

    def on_execute(self):
        with self.memoize_stage.first_step:
            self.set_info('Creating Docker Image')
            self.build_image()
        with self.memoize_stage.second_step:
            sub_task = self.find(id=self.Context.sub_task_id).first()
            if sub_task.status not in ctt.Status.Group.SUCCEED:
                raise common.errors.TaskFailure("Subtask is failed with status {}".format(sub_task.status))
            self.auth = yaml.load(sdk2.Vault.data(SEDEM_TOKEN_OWNER, SEDEM_TOKEN_NAME))['docker']
            self.set_info('Generating Porto Layer')
            self.publish_porto_layer()
