import logging
import os
import shutil
import subprocess

from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.maps.common.latest_resources import find_latest_resource
from sandbox.projects.maps.common.juggler_alerts import (
    TaskJugglerReportWithParameters
)
from sandbox.sandboxsdk import environments

TIME_FORMAT = '%H:%M:%S'


def _get_limit_timestamp(created_datetime, limit):
    # Created should be in UTC timezone (self.created in sandbox indeed is).
    limit_time = datetime.strptime(limit, TIME_FORMAT).time()
    limit_datetime = datetime.combine(created_datetime, limit_time)
    if limit_datetime < datetime.utcnow():
        limit_datetime += timedelta(days=1)
    limit_timestamp = (limit_datetime - datetime(1970, 1, 1)).total_seconds()
    return int(limit_timestamp)


class ExtraPoiExportCalcNeighborSimilaritiesExecutable(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class ExtraPoiExportRankBuildingsExecutable(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class ExtraPoiExportPushOrganizationsExecutable(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


REORDER_DATA_WITH_BLD_RANKING_QUERY_TEMPLATE = """
pragma yt.InferSchema;

INSERT INTO `{output_table}`
WITH TRUNCATE
SELECT
    coalesce(-b.priority, 0) AS priority,
    a.* WITHOUT a._other
FROM `{neighbor_similarities}` AS a
LEFT JOIN `{bld_ranking}` AS b
USING (bld_id, region)
ORDER BY priority, bld_id, region;
"""


class ExportExtraPoiBundleToNkTask(TaskJugglerReportWithParameters):
    # requrements for MULTISLOT tag
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 8 * 1024  # 8 GB
        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yql'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(TaskJugglerReportWithParameters.Parameters):
        with sdk2.parameters.RadioGroup('Environment') as environment:
            environment.values['testing'] = environment.Value('testing', default=True)
            environment.values['stable'] = environment.Value('stable')

        owner = 'MAPS'
        kill_timeout = 3 * 24 * 60 * 60  # s

        with sdk2.parameters.Group('Task parameters') as main_parameters:
            yt_vault = sdk2.parameters.Vault('YT vault', required=True)
            yql_vault = sdk2.parameters.Vault('YQL vault', required=True)
            tvm_secret = sdk2.parameters.YavSecret('TVM secret', required=True)

            with sdk2.parameters.Group('Data preprocessing parameters') as preprocessing_parameters:
                use_custom_neighbor_similarities_table = sdk2.parameters.Bool(
                    'use custom neighbor similarities table',
                    default=False
                )
                with use_custom_neighbor_similarities_table.value[True]:
                    neighbor_similarities_table = sdk2.parameters.String(
                        'neighbor_similarities table',
                        required=True,
                    )
                with use_custom_neighbor_similarities_table.value[False]:
                    calc_neighbor_similarities_executable = sdk2.parameters.Resource(
                        'Sandbox resource ID of the executable to calc neighbor_similarities table',
                        resource_type=ExtraPoiExportCalcNeighborSimilaritiesExecutable,
                        required=False
                    )
                    extra_poi_export_table = sdk2.parameters.String(
                        'Extra POI export table',
                        required=True,
                        default='//home/maps/poi/extra_poi_nyak_export/stable/full_export_data/latest'
                    )
                    nyak_mapping_table = sdk2.parameters.String(
                        'nyak mapping table',
                        required=True,
                        default='//home/altay/db/export/current-state/snapshot/nyak_mapping'
                    )
                    sprav_signals_table = sdk2.parameters.String(
                        'Sprav signals table',
                        required=True,
                        default='//home/altay/db/clusterization/state-current/clustered'
                    )
                    ymapsdf_folder = sdk2.parameters.String(
                        'ymapsdf folder',
                        required=True,
                        default='//home/maps/core/garden/stable/ymapsdf/latest'
                    )
                    include_polygons_table = sdk2.parameters.String(
                        'YT table with polygons to include buildings into upload',
                        required=False
                    )
                    exclude_polygons_table = sdk2.parameters.String(
                        'YT table with polygons to exclude from upload buildings. Does not affect geoproduct buildings',
                        required=False
                    )
                    neighbor_similarities_table_dir = sdk2.parameters.String(
                        'yt dir to store calculated neighbor_similarities table',
                        required=True,
                        default='//home/maps/poi/extra_poi_nyak_export/stable/neighbor_similarities'
                    )
                    with sdk2.parameters.RadioGroup('Filter buildings type') as filter_buildings_type:
                        filter_buildings_type.values.geoproduct = \
                            filter_buildings_type.Value(value='geoproduct', default=True)
                        filter_buildings_type.values.polygons = \
                            filter_buildings_type.Value(value='polygons')
                        filter_buildings_type.values.any = \
                            filter_buildings_type.Value(value='any')

            with sdk2.parameters.Group('Preupload ranking parameters') as bld_ranking_parameters:
                rank_buildings = sdk2.parameters.Bool(
                    'reorder buildings before upload',
                    default=False,
                )
                with rank_buildings.value[True]:
                    use_custom_bld_ranking_table = sdk2.parameters.Bool(
                        'use custom bld ranking table',
                        default=False,
                    )
                    with use_custom_bld_ranking_table.value[True]:
                        bld_ranking_table = sdk2.parameters.String(
                            'bld_ranking table',
                            required=True,
                        )
                    with use_custom_bld_ranking_table.value[False]:
                        rank_buildings_executable = sdk2.parameters.Resource(
                            'Sandbox resource ID of the executable to calc bld_ranking table',
                            resource_type=ExtraPoiExportRankBuildingsExecutable,
                            required=False,
                        )
                        bld_ranking_dir = sdk2.parameters.String(
                            'yt dir to store bld_ranking table',
                            required=True,
                            default='//home/maps/poi/extra_poi_nyak_export/stable/bld_ranking'
                        )

            with sdk2.parameters.Group('Upload process parameters') as upload_parameters:
                push_organizations_executable = sdk2.parameters.Resource(
                    'Sandbox resource ID of the executable to push organizations to nmaps',
                    resource_type=ExtraPoiExportPushOrganizationsExecutable,
                    required=False
                )
                indoor_hypotheses_dir = sdk2.parameters.String(
                    'indoor hypotheses directory',
                    required=True,
                    default='//home/maps/poi/extra_poi_nyak_export/stable/created_indoor_hypotheses'
                )
                statistics_dir = sdk2.parameters.String(
                    'yt dir to store run metrics',
                    required=True,
                    default='//home/maps/poi/extra_poi_nyak_export/stable/statistics'
                )
                upload_limit = sdk2.parameters.Integer(
                    'Upload will stop after building on which this limit was reached (moderated uploads limit)',
                    required=False
                )
                ranking_global_upload_limit = sdk2.parameters.Integer(
                    'Ranking will stop after building on which this limit was reached (overall ranking limit)',
                    required=False
                )
                upload_global_upload_limit = sdk2.parameters.Integer(
                    'Upload will stop after building on which this limit was reached (overall uploads limit)',
                    required=False
                )
                ranking_time_limit = sdk2.parameters.String(
                    'UTC time when dry run should be stopped (e.g. "07:00:00")',
                    required=False
                )
                upload_time_limit = sdk2.parameters.String(
                    'UTC time when upload should be stopped (e.g. "09:00:00")',
                    required=False
                )
                dry_run = sdk2.parameters.Bool(
                    'Do everything except uploading to NMaps and creating indoor hypotheses',
                    default=False
                )

    def _add_limits(self, cmd, upload_limit, time_limit):
        if upload_limit:
            cmd.extend(['--global-upload-limit', str(upload_limit)])
        if time_limit:
            timestamp = _get_limit_timestamp(self.created, str(time_limit))
            cmd.extend(['--timestamp-limit', str(timestamp)])

    def _load_executable(self, resource_id, name):
        resource = sdk2.Resource[resource_id]
        resource_data = sdk2.ResourceData(resource)
        executable_path = './{}'.format(name)
        shutil.copyfile(str(resource_data.path), executable_path)
        subprocess.check_call('chmod +x {}'.format(executable_path), shell=True)

    def _run_calc_neighbor_similarities_stage(self, ytc):
        with self.memoize_stage.calc_similarities:
            self.Context.start_time = datetime.now().strftime('%Y-%m-%dT%H:%M:%S')

            if self.Parameters.use_custom_neighbor_similarities_table:
                self.Context.similarities_table = str(self.Parameters.neighbor_similarities_table)
            else:
                self.Context.similarities_table = ytc.ypath_join(
                    str(self.Parameters.neighbor_similarities_table_dir),
                    self.Context.start_time,
                )
                self._load_executable(
                    self.calc_neighbor_similarities_executable,
                    'calc_neighbors'
                )
                cmd = [
                    './calc_neighbors',
                    '--extra-poi-nk-export', str(self.Parameters.extra_poi_export_table),
                    '--nyak-mapping', str(self.Parameters.nyak_mapping_table),
                    '--sprav-signals', str(self.Parameters.sprav_signals_table),
                    '--ymapsdf-folder', str(self.Parameters.ymapsdf_folder),
                    '--filter-buildings-type', self.Parameters.filter_buildings_type,
                    '--output-table', self.Context.similarities_table
                ]
                if self.Parameters.include_polygons_table:
                    cmd += [
                        '--include-polygons', self.Parameters.include_polygons_table
                    ]
                if self.Parameters.exclude_polygons_table:
                    cmd += [
                        '--exclude-polygons', self.Parameters.exclude_polygons_table
                    ]

                logging.info('Running command %s', ' '.join(cmd))
                with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('calc similarities')) as pl:
                    subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)

    def _reorder_data_with_bld_ranking(self, ytc):
        from yql.api.v1.client import YqlClient

        yql = YqlClient(db='hahn', token=self.Parameters.yql_vault.data())
        result = ytc.create_temp_table()
        query = REORDER_DATA_WITH_BLD_RANKING_QUERY_TEMPLATE.format(
            bld_ranking=self.Context.bld_ranking_table,
            neighbor_similarities=self.Context.similarities_table,
            output_table=result,
        )
        request = yql.query(query, syntax_version=1)
        request.run()
        request.get_results(wait=True)
        if not request.is_success:
            raise errors.TaskError('Error applying building ordering')
        self.Context.similarities_table = result

    def _run_bld_ranking_stage(self, ytc):
        with self.memoize_stage.bld_ranking:
            if self.Parameters.use_custom_bld_ranking_table:
                self.Context.bld_ranking_table = str(self.Parameters.bld_ranking_table)
            else:
                self._load_executable(
                    self.push_organizations_executable,
                    'upload_poi'
                )
                dry_run_statistics = ytc.create_temp_table()
                cmd = [
                    './upload_poi',
                    '--extra-poi-export-table', self.Context.similarities_table,
                    '--indoor-hypotheses-dir', str(self.Parameters.indoor_hypotheses_dir),
                    '--statistics-table', dry_run_statistics,
                    '--dry-run',
                ]
                self._add_limits(
                    cmd,
                    self.Parameters.ranking_global_upload_limit,
                    self.Parameters.ranking_time_limit
                )

                logging.info('Running command %s', ' '.join(cmd))
                with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('upload poi dry run')) as pl:
                    subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)

                self.Context.bld_ranking_table = ytc.ypath_join(
                    str(self.Parameters.bld_ranking_dir),
                    self.Context.start_time,
                )
                self._load_executable(
                    self.rank_buildings_executable,
                    'rank_buildings'
                )
                cmd = [
                    './rank_buildings',
                    '--dry-run-stats-table', dry_run_statistics,
                    '--neighbor-similarities-table', self.Context.similarities_table,
                    '--output-table', self.Context.bld_ranking_table,
                ]
                logging.info('Running command %s', ' '.join(cmd))
                with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('rank buildings')) as pl:
                    subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)

            self._reorder_data_with_bld_ranking(ytc)

    def _run_upload_organizations_stage(self):
        with self.memoize_stage.upload_poi:
            self._load_executable(
                self.push_organizations_executable,
                'upload_poi'
            )
            cmd = [
                './upload_poi',
                '--extra-poi-export-table', self.Context.similarities_table,
                '--indoor-hypotheses-dir', str(self.Parameters.indoor_hypotheses_dir),
                '--statistics-dir', str(self.Parameters.statistics_dir),
            ]
            self._add_limits(
                cmd,
                self.Parameters.upload_global_upload_limit,
                self.Parameters.upload_time_limit
            )
            if self.Parameters.upload_limit:
                cmd.extend(['--upload-limit', str(self.Parameters.upload_limit)])
            if self.Parameters.dry_run:
                cmd.append('--dry-run')

            logging.info('Running command %s', ' '.join(cmd))
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('upload poi')) as pl:
                subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)

    def ensure_latest_resources_used(self):
        self.calc_neighbor_similarities_executable = self.Parameters.calc_neighbor_similarities_executable
        if not self.calc_neighbor_similarities_executable:
            self.calc_neighbor_similarities_executable = find_latest_resource(
                ExtraPoiExportCalcNeighborSimilaritiesExecutable, self.Parameters.environment)

        self.rank_buildings_executable = self.Parameters.rank_buildings_executable
        if not self.rank_buildings_executable:
            self.rank_buildings_executable = find_latest_resource(
                ExtraPoiExportRankBuildingsExecutable, self.Parameters.environment)

        self.push_organizations_executable = self.Parameters.push_organizations_executable
        if not self.push_organizations_executable:
            self.push_organizations_executable = find_latest_resource(
                ExtraPoiExportPushOrganizationsExecutable, self.Parameters.environment)

        logging.info(
            'Working in %s environment', self.Parameters.environment)
        logging.info(
            'Using ExtraPoiExportCalcNeighborSimilaritiesExecutable: %s',
            self.calc_neighbor_similarities_executable.id)
        logging.info(
            'Using ExtraPoiExportRankBuildingsExecutable: %s',
            self.rank_buildings_executable.id)
        logging.info(
            'Using ExtraPoiExportPushOrganizationsExecutable: %s',
            self.push_organizations_executable.id)

    def on_execute(self):
        import yt.wrapper as yt

        yt.config['proxy']['url'] = 'hahn'
        yt.config['token'] = self.Parameters.yt_vault.data()

        os.environ['YT_TOKEN'] = self.Parameters.yt_vault.data()
        os.environ['YQL_TOKEN'] = self.Parameters.yql_vault.data()
        os.environ['TVM_SECRET'] = self.Parameters.tvm_secret.data()['client_secret']

        self.ensure_latest_resources_used()

        self._run_calc_neighbor_similarities_stage(yt)

        if self.Parameters.rank_buildings:
            self._run_bld_ranking_stage(yt)

        self._run_upload_organizations_stage()
