import datetime
import logging
import os
import shutil
import subprocess
import time

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.common.types.task import Status
from sandbox.sandboxsdk import environments

from sandbox.projects.maps.common.juggler_alerts import (
    TaskJugglerReportWithParameters
)


EXTRACT_METRIKA_TASK_TIMEOUT_SECONDS = 12 * 60 * 60  # 12 hours


class ExtractMetrikaExecutable(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class ExtractMetrikaTask(sdk2.Task):
    # requrements for MULTISLOT tag
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024  # 1 GB

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = EXTRACT_METRIKA_TASK_TIMEOUT_SECONDS
        yt_vault = sdk2.parameters.Vault('YT vault', required=True)
        yql_vault = sdk2.parameters.Vault('YQL vault', required=True)

        executable = sdk2.parameters.Resource(
            'Sandbox resource ID of the executable to extract Metrika data',
            resource_type=ExtractMetrikaExecutable,
            required=True
        )
        date = sdk2.parameters.String(
            'Date of Metrika export',
            required=True
        )
        output_dir = sdk2.parameters.String(
            'YT-directory with output tables',
            required=True
        )

    def _load_executable(self, resource_id, name):
        resource = sdk2.Resource[resource_id]
        resource_data = sdk2.ResourceData(resource)
        executable_path = './{}'.format(name)
        shutil.copyfile(str(resource_data.path), executable_path)
        subprocess.check_call('chmod +x {}'.format(executable_path), shell=True)

    def on_execute(self):
        self._load_executable(self.Parameters.executable, 'executable')
        os.environ['YT_TOKEN'] = self.Parameters.yt_vault.data()
        os.environ['YQL_TOKEN'] = self.Parameters.yql_vault.data()

        cmd = [
            './executable',
            '--date', self.Parameters.date,
            '--output-dir', self.Parameters.output_dir
        ]

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('executable')) as pl:
            subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)


class ExtractMetrika(TaskJugglerReportWithParameters):
    class Requirements(sdk2.Task.Requirements):
        environments = [environments.PipEnvironment('yandex-yt')]

    class Parameters(TaskJugglerReportWithParameters.Parameters):
        yt_vault = sdk2.parameters.Vault('YT vault', required=True)
        yql_vault = sdk2.parameters.Vault('YQL vault', required=True)

        executable = sdk2.parameters.Resource(
            'Sandbox resource ID of the executable to extract Metrika data',
            resource_type=ExtractMetrikaExecutable,
            required=True
        )
        max_days = sdk2.parameters.Integer(
            'Maximum number of last days for which data will be extracted if they do not exist',
            default=14
        )
        output_dir = sdk2.parameters.String(
            'YT-directory with output tables',
            required=True
        )

    def _make_dates_list(self):
        import yt.wrapper as yt
        yt.config['proxy']['url'] = 'hahn'
        yt.config['token'] = self.Parameters.yt_vault.data()

        tables = set(yt.list(self.Parameters.output_dir, absolute=False))
        result = []
        utc_now = datetime.datetime.utcfromtimestamp(time.time())
        for day in range(1, self.Parameters.max_days + 1):
            date = (utc_now - datetime.timedelta(days=day)).strftime('%Y-%m-%d')
            if date not in tables:
                result.append(date)
        return result

    def _check_subtasks(self):
        if not all(
            task.status == Status.SUCCESS
            for task in self.find(parent_id=self.id)
        ):
            raise TaskError('Child task failed')

    def on_execute(self):
        with self.memoize_stage.extract_by_days:
            dates = self._make_dates_list()
            extract_metrika_tasks_ids = []
            for date in dates:
                logging.info('Extract from metrika, date %s', date)
                task = ExtractMetrikaTask(
                    self,
                    description='Extract from metrika, date {}'.format(date),
                    yt_vault=self.Parameters.yt_vault,
                    yql_vault=self.Parameters.yql_vault,
                    executable=self.Parameters.executable,
                    output_dir=self.Parameters.output_dir,
                    date=date
                )
                task.enqueue()
                extract_metrika_tasks_ids.append(task.id)

            raise sdk2.WaitTask(
                extract_metrika_tasks_ids,
                Status.Group.FINISH + Status.Group.BREAK,
                wait_all=True,
                timeout=EXTRACT_METRIKA_TASK_TIMEOUT_SECONDS
            )

        self._check_subtasks()
