import json
import os
import re
import yaml

from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.projects.maps.common import utils
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.common.vcs.arc import Arc

SEDEM_TOKEN_OWNER = "MAPS"
SEDEM_OAUTH = "SEDEM_TOKEN"

REVIEW_CHANGELOG_HEADER = """
| Revision | Author | Summary |
| -- | -- | -- |
"""
REVIEW_CHANGELOG_LINE_FORMAT = '| [**r{revision}**](https://a.yandex-team.ru/arc/commit/{revision}) | @{author} | {summary} |'
FILE_CHANGELOG_LINE_FORMAT = '**r{revision}** @{author} {summary}\n'


def format_changelog(changelog, line_format):
    result = []
    for revision, content in sorted(changelog.items(), reverse=True, key=lambda item: item[0]):
        result.append(line_format.format(revision=revision,
                                         author=content['author'],
                                         summary=content['message'].split('\n\n')[0])) # ignore REVIEW:id or NOTE:...
    return '\n'.join(result)


class MapsBaseImageUpdater(sdk2.Task):

    class Requirements(sdk2.Requirements):
        client_tags = Tag.GENERIC
        disk_space = 10 * 1024  # 10GB

    class Parameters(sdk2.Parameters):
        baseimage_revision = sdk2.parameters.Integer('SVN revision of baseimage', required=True)
        with sdk2.parameters.RadioGroup("Review debug level") as debug_level:
            debug_level.values.NONE = debug_level.Value("No debug", default=True)
            debug_level.values.NO_PUBLISH = "Do not publish review"
            debug_level.values.DRY_RUN = "Do not create review and do not commit"
        allow_downgrade = sdk2.parameters.Bool('Allow downgrade image version')
        marker_file_path = sdk2.parameters.String('Path to marker file to write merge stamp', required=True, default='maps/infra/baseimage/current_version.json')
        changelog_paths = sdk2.parameters.List("Changelog observed paths", required=True, default=
            ['maps/infra/baseimage', 'maps/infra/yacare', 'maps/infra/roquefort', 'maps/infra/ecstatic/common',
             'maps/infra/ecstatic/client', 'maps/infra/ecstatic/config', 'maps/infra/ecstatic/tool',
             'maps/infra/ecstatic/ymtorrent', 'maps/infra/tvm2lua', 'maps/infra/ratelimiter2/core',
             'maps/infra/ratelimiter2/common', 'maps/infra/ratelimiter2/proxy', 'maps/infra/ratelimiter2/plugin'])
        changelog_ignored_paths = sdk2.parameters.List("Changelog ignored paths", required=True, default=
            ['maps/infra/baseimage/current_version.json', 'maps/infra/baseimage/README.md',
             'maps/infra/baseimage/CHANGELOG.md', 'maps/infra/baseimage/sedem_config',
             'maps/infra/ecstatic/tool/package', 'maps/infra/baseimage/release_tools'])
        with sdk2.parameters.Output:
            review_index = sdk2.parameters.Integer("Generated PR index", required=False)

    def on_execute(self):
        with arcadia_sdk.mount_arc_path('arcadia-arc:/#trunk') as arcadia:
            self.arcadia = arcadia
            self.execute(arcadia)

    def _is_commit_ignored(self, commit):
        for path in commit['names']:
            if len(filter(lambda x: path['path'].startswith(x), self.Parameters.changelog_ignored_paths)) == 0:
                return False
        return True

    def generate_changelog(self, last_released, new_revision):
        arc = Arc()
        changelog = {}
        for path in self.Parameters.changelog_paths:
            log = arc.log(self.arcadia, path=path, start_commit='r{}'.format(last_released),
                          end_commit='r{}'.format(new_revision), as_dict=True, name_only=True)
            for item in log:
                if not self._is_commit_ignored(item):
                    changelog[int(item['revision'])] = {'message': item['message'], 'author': item['author']}
        return changelog

    def execute(self, arcadia):
        # NB:
        #   devtools/contrib/piglet is using maps base docker image.
        #   Such usage is approved by geo-infra@
        top_level_dirs = ('devtools', 'maps')
        for top_level_dir in top_level_dirs:
            for dir, subdirs, files in os.walk(os.path.join(arcadia, top_level_dir)):
                for file in files:
                    if file != 'Dockerfile':
                        continue
                    self.update_docker_base_image(os.path.join(dir, file), self.Parameters.baseimage_revision)

        with open(os.path.join(arcadia, self.Parameters.marker_file_path)) as file:
            previous_baseimage_revision = int(json.load(file)['svn_revision'])

        with open(os.path.join(arcadia, self.Parameters.marker_file_path), "w") as file:
            json.dump({'svn_revision': self.Parameters.baseimage_revision}, file)

        changelog = self.generate_changelog(previous_baseimage_revision, self.Parameters.baseimage_revision)

        if self.Parameters.debug_level != 'DRY_RUN':
            message = 'Update base image version: to {}'.format(self.Parameters.baseimage_revision)
            description = ("\n_Changelog_\n" +
                          REVIEW_CHANGELOG_HEADER +
                          format_changelog(changelog, REVIEW_CHANGELOG_LINE_FORMAT))
            output = utils.arc_commit(
                arcadia,
                paths=[os.path.join(arcadia, top_level_dir) for top_level_dir in top_level_dirs],
                message=message,
                description=description)
            self.set_info(output)
            review_url = output.split('\n')[-1]
            review_index = review_url.split('/')[-1]
            self.set_info('Review {review_url} was created'.format(**locals()), do_escape=False)
            self.Parameters.review_index = review_index

    def update_docker_base_image(self, filename, revision):
        with open(filename, 'r') as file:
            data = file.read()

        # Updating base image version and default name ('maps/core-base' to 'maps/core-base-trusty')
        # Lookup line "FROM registry.yandex.net/maps/<current_name>:<current_version>" (ignore commented)
        # Optional line "#autoupdate: prestable" defines staging
        match = re.match(r'.*?(?:^[ \t]*#[ \t]*autoupdate:[ \t]*(\w*)[ \t]*\n)?'
                         r'(?:^[ \t]*FROM[ \t]+registry.yandex.net/maps/(core-base(?:-\w+)?):(\w+)[ \t]*$)',
                         data,
                         re.DOTALL | re.MULTILINE)
        if match:
            staging = match.group(1)

            current_name = match.group(2)
            if current_name == 'core-base':
                new_name = 'core-base-trusty'  # new default base image name
            else:
                new_name = current_name  # leave non-default image name, e.g. core-base-bionic

            current_version = match.group(3)

            if current_version == revision:
                self.set_info('Already correct base image version {current_version} in {filename}'.format(**locals()))
                return

            if not self.Parameters.allow_downgrade:
                try:
                    if revision < int(current_version):
                        self.set_info('Skip downgrading base image version in {filename}'.format(**locals()))
                        return
                except ValueError:
                    pass
            # Replace "FROM registry.yandex.net/maps/<current_name>:<current_version>"
            # with "FROM registry.yandex.net/maps/<new_name>:<new_version>
            data = re.sub(r'^([ \t]*FROM([ \t]+)registry.yandex.net/maps/)(core-base(?:-\w+)?):\w+',
                          r'\1{}:{}'.format(new_name, revision),
                          data,
                          flags=re.MULTILINE)
            with open(filename,'w') as file:
                file.write(data)
            self.set_info('Changing base image from {current_name}:{current_version} to '
                          '{new_name}:{revision} in {filename}'.format(**locals()))
