from sandbox import sdk2

from sandbox.sdk2.vcs.svn import Svn
from sandbox.common.errors import TemporaryError, TaskFailure
import sandbox.common.types.resource as ctr

import os
import json
import signal
import time
import logging


class LogCollector(logging.Filter):
    def __init__(self):
        self.stdout = []
        self.stderr = []
        super(LogCollector, self).__init__()

    def filter(self, record):
        if record.levelno == logging.INFO:
            self.stdout.append(record.msg)
            record.name = 'stdout'
        elif record.levelno == logging.ERROR:
            self.stderr.append(record.msg)
            record.name = 'stderr'
        record.levelname = 'INFO'
        record.levelno = logging.INFO
        return True


class MapsBinaryBaseTask(sdk2.Task):
    """
    This task runs arbitrary binary (loaded as Sandbox resource) in Sandbox and implements staging.

    HowTo:
    1) Write a regular binary which takes all arguments as command line options and arguments,
       options should not contain underscores
    2.1) Build your binary locally and upload it to sandbox using `ya upload <your binary>`,
         then remember given resource_id
    2.2) Or run YA_MAKE Sandbox task with the path to the source of your binary,
         then remember resource_id of BINARY_OUTPUT
    3) Do not forget to extend ttl of new resource, if you need (`Make Important` button in Sandbox GUI)
    4) If you need staging, put simple json file in Arcadia with dict {"environment": resource_id}
    5.0) Inherit from this binary your new Sandbox task
    5.1) Add same options and parameters as in your binary, you may use helpers options_from_parameters and
         args_from_parameters to convert Sandbox parameters to commandline options and args
    5.2) You may use register_installations helper to make drop-box parameter for staging
    5.3) Add default name to binary_name parameter

    You may use MapsNotificationCounter task as an example

    By the way, there is also MapsBinaryTask which inherits this task and allows to run any binary
    with arbitrary options and parameters
    """

    RETRIABLE_EXITCODE = 100
    BEFORE_TIMEOUT_STOP_SECONDS = 20

    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space =  1024  # 1GB
        ram = 2048  # 2GB

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        @staticmethod
        def register_installations(installations, default='testing'):
            with sdk2.parameters.String('Installation', multiline=True) as installation:
                for key in installations:
                    installation.values[key] = installation.Value(value=key, default=(key==default))
                return installation

        binary = sdk2.parameters.Resource('Resource with binary or None for auto-detect', required=False)
        version_file_path = sdk2.parameters.String('Path to version file', required=False)
        binary_type = sdk2.parameters.String('Binary type. Last released version will be used', required=False)
        installation = sdk2.parameters.String('Installation. Stable by default', required=False)
        binary_name = sdk2.parameters.String('Binary name', required=False)

    def on_execute(self):
        self.run_binary()

    def _load_binary(self):
        resource = self.Parameters.binary
        installation = self.Parameters.installation or 'stable'
        binary_type = self.Parameters.binary_type
        if not resource:
            if binary_type:
                staging_resource = sdk2.Resource.find(type=binary_type, state=ctr.State.READY,
                                              attrs={'released': installation}).first()
                stable_resource = sdk2.Resource.find(type=binary_type, state=ctr.State.READY,
                                                     attrs={'released': 'stable'}).first()
                if not staging_resource or (stable_resource and stable_resource.id > staging_resource.id):
                    resource = stable_resource
                else:
                    resource = staging_resource
            else:
                resource_versions = json.loads(Svn.cat(os.path.join(
                    'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/',
                    self.Parameters.version_file_path)))
                resource_id = resource_versions[installation]
                resource = sdk2.Resource.find(id=resource_id).first()
        self.set_info('Using binary from: ' + str(resource))
        binary_data = sdk2.ResourceData(resource)
        binary_path = binary_data.path  # ya upload case for development
        if not binary_path.is_file():
            binary_path = binary_path.joinpath('bin/' + self.Parameters.binary_name)  # YA_MAKE case
        return binary_path

    def options_from_parameters(self, options_list):
        filter_empty = lambda d: {k: v for k, v in d.iteritems() if v}
        return filter_empty({option: getattr(self.Parameters, option, None) for option in options_list})

    def args_from_parameters(self, args_list):
        return filter(None, (getattr(self.Parameters, arg, None) for arg in args_list))

    def run_binary(self, options={}, subcmd=None, subcmd_options={}, arguments=[], env_options={}, cwd=None):
        def dict_to_args(options):
            args = []
            for key, value in options.iteritems():
                if value is not None:
                    key_str = '--' + key.lstrip('-').replace('_', '-')
                    if isinstance(value, bool):
                        args.append(key_str)
                    elif isinstance(value, list) or isinstance(value, tuple):
                        for arg in value:
                            args += [key_str, str(arg)]
                    else:
                        args += [key_str, str(value)]
            return args

        os.environ['MAPS_BINARY_TASK_LOGS_DIR'] = str(self.log_path())
        for option_name, option_value in env_options.iteritems():
            os.environ[option_name] = option_value

        logger = logging.getLogger('binary')
        logger.setLevel(logging.INFO)
        log_collector = LogCollector()
        logger.addFilter(log_collector)

        binary_path = self._load_binary()
        with sdk2.helpers.ProcessRegistry:
            try:
                if subcmd:
                    args = [str(binary_path)] + dict_to_args(options) + \
                           [str(subcmd)] + dict_to_args(subcmd_options) + \
                           list(arguments)
                else:
                    args = [str(binary_path)] + dict_to_args(options) + \
                           list(arguments)
                with sdk2.helpers.ProcessLog(logger=logger, stdout_level=logging.INFO, stderr_level=logging.ERROR) as pl:
                    sdk2.helpers.subprocess.check_call(args, stdout=pl.stdout, stderr=pl.stderr, cwd=cwd)
                self.set_info('\n'.join(log_collector.stdout))
                self.set_info('stderr:\n'+'\n'.join(log_collector.stderr))
            except sdk2.helpers.ProcessLog.CalledProcessError as e:
                error_str = 'exitcode: {code}\nstdout:\n{stdout}\nstderr:\n{stderr}'.format(
                    code=e.returncode,
                    stdout='\n'.join(log_collector.stdout),
                    stderr='\n'.join(log_collector.stderr))
                if e.returncode == self.RETRIABLE_EXITCODE:
                    raise TemporaryError(error_str)
                else:
                    raise TaskFailure(error_str)
            finally:
                logger.removeFilter(log_collector)

    def on_before_timeout(self, seconds):
        if seconds <= self.BEFORE_TIMEOUT_STOP_SECONDS:
            for process in sdk2.helpers.ProcessRegistry:
                try:
                    self.set_info('Sending SIGINT to binary')
                    os.kill(process.pid, signal.SIGINT)
                    time.sleep(1)
                    self.set_info('Sending SIGTERM to binary')
                    os.kill(process.pid, signal.SIGTERM)
                    time.sleep(1)
                    if seconds <= min(self.timeout_checkpoints()):
                        self.set_info('Sending SIGKILL to binary')
                        os.kill(process.pid, signal.SIGKILL)
                except OSError:
                    continue
        super(MapsBinaryBaseTask, self).on_before_timeout(seconds)

    def timeout_checkpoints(self):
        return [self.BEFORE_TIMEOUT_STOP_SECONDS / 2, self.BEFORE_TIMEOUT_STOP_SECONDS]
