from sandbox import sdk2

from sandbox.sdk2.vcs.svn import Svn
from sandbox.common.errors import TemporaryError, TaskFailure

import os
import json
import signal
import time
import logging


class LogCollector(logging.Filter):
    def __init__(self):
        self.stdout = []
        self.stderr = []
        super(LogCollector, self).__init__()

    def filter(self, record):
        if record.levelno == logging.INFO:
            self.stdout.append(record.msg)
            record.name = 'stdout'
        elif record.levelno == logging.ERROR:
            self.stderr.append(record.msg)
            record.name = 'stderr'
        record.levelname = 'INFO'
        record.levelno = logging.INFO
        return True


class MapsBinaryTaskTest(sdk2.Task):

    RETRIABLE_EXITCODE = 100
    BEFORE_TIMEOUT_STOP_SECONDS = 20

    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space =  1024  # 1GB

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        @staticmethod
        def register_installations(installations, default='testing'):
            with sdk2.parameters.String('Installation', multiline=True) as installation:
                for key in installations:
                    installation.values[key] = installation.Value(value=key, default=(key==default))
                return installation

        binary = sdk2.parameters.Resource('Resource with binary or None for auto-detect', required=False)
        version_file_path = sdk2.parameters.String('Path to version file', required=False)
        installation = sdk2.parameters.String('Installation', required=False)
        binary_name = sdk2.parameters.String('Binary name', required=False)

        args = sdk2.parameters.List('Args for binary')
        options = sdk2.parameters.Dict('Options for binary')
        subcmd = sdk2.parameters.String('Subcmd', required=False)
        subcmd_options = sdk2.parameters.Dict('Options for subcmd', required=False)
        vault_options = sdk2.parameters.Dict('Vault options for binary')
        env_options = sdk2.parameters.Dict('Environment options for binary')
        vault_env_options = sdk2.parameters.Dict('Vault environment options for binary')

    def _load_binary(self):
        resource = self.Parameters.binary
        installation = self.Parameters.installation
        if not resource and installation:
            resource_versions = json.loads(Svn.cat(os.path.join(
                'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/',
                self.Parameters.version_file_path)))
            resource_id = resource_versions[installation]
            resource = sdk2.Resource.find(id=resource_id).first()
        self.set_info('Using binary from: ' + str(resource))
        binary_data = sdk2.ResourceData(resource)
        binary_path = binary_data.path  # ya upload case for development
        if not binary_path.is_file():
            binary_path = binary_path.joinpath('bin/' + self.Parameters.binary_name)  # YA_MAKE case
        return binary_path

    def run_binary(self, options={}, subcmd=None, subcmd_options={}, arguments=[], env_options={}):
        def dict_to_args(options):
            args = []
            for key, value in options.iteritems():
                if value:
                    key_str = '--' + key.lstrip('-').replace('_', '-')
                    if isinstance(value, bool):
                        args.append(key_str)
                    elif isinstance(value, list) or isinstance(value, tuple):
                        for arg in value:
                            args += [key_str, str(arg)]
                    else:
                        args += [key_str, str(value)]
            return args

        for option_name, option_value in env_options.iteritems():
            os.environ[option_name] = option_value

        logger = logging.getLogger('binary')
        logger.setLevel(logging.INFO)
        log_collector = LogCollector()
        logger.addFilter(log_collector)

        binary_path = self._load_binary()
        with sdk2.helpers.ProcessRegistry:
            try:
                if subcmd:
                    args = [str(binary_path)] + dict_to_args(options) + \
                           [str(subcmd)] + dict_to_args(subcmd_options) + \
                           list(arguments)
                else:
                    args = [str(binary_path)] + dict_to_args(options) + \
                           list(arguments)
                with sdk2.helpers.ProcessLog(logger=logger, stdout_level=logging.INFO, stderr_level=logging.ERROR) as pl:
                    sdk2.helpers.subprocess.check_call(args, stdout=pl.stdout, stderr=pl.stderr)
                self.set_info('\n'.join(log_collector.stdout))
                self.set_info('stderr:\n'+'\n'.join(log_collector.stderr))
            except sdk2.helpers.ProcessLog.CalledProcessError as e:
                error_str = 'exitcode: {code}\nstdout:\n{stdout}\nstderr:\n{stderr}'.format(
                    code=e.returncode,
                    stdout='\n'.join(log_collector.stdout),
                    stderr='\n'.join(log_collector.stderr))
                if e.returncode == self.RETRIABLE_EXITCODE:
                    raise TemporaryError(error_str)
                else:
                    raise TaskFailure(error_str)
            finally:
                logger.removeFilter(log_collector)

    def on_before_timeout(self, seconds):
        if seconds <= self.BEFORE_TIMEOUT_STOP_SECONDS:
            for process in sdk2.helpers.ProcessRegistry:
                try:
                    self.set_info('Sending SIGINT to binary')
                    os.kill(process.pid, signal.SIGINT)
                    time.sleep(1)
                    self.set_info('Sending SIGTERM to binary')
                    os.kill(process.pid, signal.SIGTERM)
                    time.sleep(1)
                    if seconds <= min(self.timeout_checkpoints()):
                        self.set_info('Sending SIGKILL to binary')
                        os.kill(process.pid, signal.SIGKILL)
                except OSError:
                    continue
        super(MapsBinaryBaseTask, self).on_before_timeout(seconds)

    def timeout_checkpoints(self):
        return [self.BEFORE_TIMEOUT_STOP_SECONDS / 2, self.BEFORE_TIMEOUT_STOP_SECONDS]

    def on_execute(self):
        options = self.options_from_vault(self.Parameters.vault_options)
        options.update(self.Parameters.options)
        env_options = self.options_from_vault(self.Parameters.vault_env_options)
        env_options.update(self.Parameters.env_options)
        self.run_binary(
            options=self.convert_empty_to_true(options),
            arguments=self.Parameters.args,
            subcmd=self.Parameters.subcmd,
            subcmd_options=self.Parameters.subcmd_options,
            env_options=env_options)

    @staticmethod
    def options_from_vault(vault_options_dict):
        options = {}
        for option, vault_item in vault_options_dict.iteritems():
            owner, vault = vault_item.split(':') if ':' in vault_item else (None, vault_item)
            options[option] = sdk2.Vault.data(owner, vault)
        return options

    @staticmethod
    def convert_empty_to_true(options):
        return {k: True if v == '' else v for k, v in options.iteritems()}
