import logging
import os

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sdk_subprocess
import subprocess
from sandbox.projects.maps.common.ecstatic_bin import MapsEcstaticToolMixin
from sandbox.common.errors import TaskFailure
from sandbox.common.telegram import TelegramBot
from sandbox.sandboxsdk import environments


TVM_IDS = {
    'testing': 2018420,
    'stable': 2018422,
}
TVM_SEC_UUIDS = {
    'testing': 'sec-01e0q9xznk7j2qjd196w49drf3',
    'stable': 'sec-01e0q9z2q3wk1fbhzx7cek4kvb',
}


# Resources
class MapsEdgeSeqStatBuilder(sdk2.Resource):
    """
    Build edge sequences statistics executable
    maps/libs/jams/edge_seq_stat/build_edge_seq_stat
    """
    executable = True


class MapsBuildEdgeSeqStat(MapsEcstaticToolMixin, sdk2.Task):
    """
    Task for build edge sequences statistics
    """
    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment("yandex-yt"),
        )

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 12 * 60 * 60

        with sdk2.parameters.Group('Sandbox resources') as resources_group:
            maps_edge_seq_stat_builder = sdk2.parameters.Resource(
                'Build edge sequences statistics executable',
                resource_type=MapsEdgeSeqStatBuilder,
                required=True
            )

            maps_routes_popularity_builder = sdk2.parameters.Resource(
                'Build routes popularity statistics executable',
                resource_type=MapsEdgeSeqStatBuilder,
                required=False
            )

        with sdk2.parameters.Group('Statistics parameters') as stat_group:
            stat_date_count = sdk2.parameters.Integer(
                'Number of days in period for tracks',
                required=True,
                default=14
            )
            stat_key_sizes = sdk2.parameters.List(
                'Sizes of edge sequence',
                required=True,
                value_type=sdk2.parameters.Integer,
                default=[15, 10, 3, 2]
            )
            stat_threshold = sdk2.parameters.Integer(
                'Threshold for tracks count',
                required=True,
                default=140
            )
            stat_user_routes_threshold = sdk2.parameters.Integer(
                'Threshold for user tracks count',
                required=True,
                default=3
            )

        with sdk2.parameters.Group('YT parameters') as yt_group:
            yt_proxy = sdk2.parameters.String(
                'YT proxy',
                required=True,
                default='hahn.yt.yandex.net'
            )
            yt_pool_tree = sdk2.parameters.String(
                'YT pool tree',
                required=False,
                default='yamaps'
            )
            yt_pool = sdk2.parameters.String(
                'YT pool',
                required=False,
                default='yamaps_dev'
            )
            yt_token = sdk2.parameters.String(
                'YT token sandbox vault record name',
                required=True,
                default='yt-token'
            )
            yt_token_owner = sdk2.parameters.String(
                'YT token sandbox vault record owner',
                required=True
            )
            yt_input_path = sdk2.parameters.String(
                'YT path to daily travel times tables',
                required=True,
                default='//home/maps/jams/production/quality-daily/travel_times'
            )
            yt_events_path = sdk2.parameters.String(
                'YT path to daily event log tables',
                required=True,
                default='//home/statistics-navi/production/event_log'
            )

        with sdk2.parameters.Group('Ecstatic parameters') as ecstatic_group:
            with sdk2.parameters.RadioGroup('Ecstatic environment') as ecstatic_environment:
                ecstatic_environment.values['development'] = ecstatic_environment.Value(value='Development', default=True)
                ecstatic_environment.values['testing'] = ecstatic_environment.Value(value='Testing')
                ecstatic_environment.values['stable'] = ecstatic_environment.Value(value='Stable')

            ecstatic_dataset = sdk2.parameters.String(
                'Ecstatic dataset',
                required=True,
                default='yandex-maps-jams-edge-sequences-statistics')

            ecstatic_branch = sdk2.parameters.String(
                "Dataset branch",
                required=True)
            with sdk2.parameters.RadioGroup('Dataset branch') as ecstatic_branch:
                ecstatic_branch.values['development'] = ecstatic_branch.Value(value='development')
                ecstatic_branch.values['testing'] = ecstatic_branch.Value(value='testing')
                ecstatic_branch.values['stable'] = ecstatic_branch.Value(value='stable', default=True)

        with sdk2.parameters.Group('Telegram parameters') as telegram_group:
            telegram_token = sdk2.parameters.String(
                'Telegram token sandbox vault record name',
                required=False
            )
            telegram_token_owner = sdk2.parameters.String(
                'Telegram token sandbox vault record owner',
                required=False
            )
            telegram_chat_id = sdk2.parameters.Integer(
                'Chat id',
                required=False,
                default=-1001465293474
            )

    def on_execute(self):
        self._prepare_yt_token()

        day = self._get_last_table(self.Parameters.yt_input_path)
        last_events_table = self._get_last_table(self.Parameters.yt_events_path)
        last_version = self._get_last_version()
        if last_version and last_version >= day:
            return
        if last_events_table < day:
            return

        dataset_dir = 'dataset'
        os.makedirs(dataset_dir)

        self._build_stat(dataset_dir, day)
        self._build_routes_popularity(dataset_dir, day)

        self._upload_ecstatic(dataset_dir, day)

    def _prepare_yt_token(self):
        logging.info('Prepare yt token...')

        yt_token = sdk2.Vault.data(
            self.Parameters.yt_token_owner,
            self.Parameters.yt_token)
        os.environ["YT_TOKEN"] = yt_token
        logging.info('Success')

    def _build_stat(self, dataset_dir, day):
        logging.info('Build edge sequences statistics...')

        output_file = os.path.join(dataset_dir, 'edge_sequence_stat.mms.2')

        executable = str(sdk2.ResourceData(self.Parameters.maps_edge_seq_stat_builder).path)
        cmd_params = [
            '--proxy', self.Parameters.yt_proxy,
            '--input', self.Parameters.yt_input_path,
            '--events', self.Parameters.yt_events_path,
            '--date-count', str(self.Parameters.stat_date_count),
            '--keySizes', ','.join(map(str, sorted(self.Parameters.stat_key_sizes, reverse=True))),
            '--threshold', str(self.Parameters.stat_threshold),
            '--date', day,
            '--output', output_file
        ]
        if self.Parameters.yt_pool_tree:
            cmd_params.extend(['--poolTree', self.Parameters.yt_pool_tree])
        if self.Parameters.yt_pool:
            cmd_params.extend(['--pool', self.Parameters.yt_pool])

        logging.info('Run command: {}'.format(' '.join([executable] + cmd_params)))
        with sdk2.helpers.ProcessLog(self, logger='maps_edge_seq_stat_builder') as l:
            sdk_subprocess.check_call([executable] + cmd_params, stdout=l.stdout, stderr=l.stderr)

        logging.info('Success')

    def _build_routes_popularity(self, dataset_dir, day):
        logging.info('Build routes popularity statistics...')

        output_file = os.path.join(dataset_dir, 'edge_sequence_stat.mms.3')

        if not self.Parameters.maps_routes_popularity_builder:
            logging.info('Build routes popularity skiped')
            return

        executable = str(sdk2.ResourceData(self.Parameters.maps_routes_popularity_builder).path)
        cmd_params = [
            '--proxy', self.Parameters.yt_proxy,
            '--input', self.Parameters.yt_input_path,
            '--events', self.Parameters.yt_events_path,
            '--date-count', str(self.Parameters.stat_date_count),
            '--threshold', str(self.Parameters.stat_threshold),
            '--date', day,
            '--output', output_file,
        ]
        if self.Parameters.yt_pool_tree:
            cmd_params.extend(['--poolTree', self.Parameters.yt_pool_tree])
        if self.Parameters.yt_pool:
            cmd_params.extend(['--pool', self.Parameters.yt_pool])

        logging.info('Run command: {}'.format(' '.join([executable] + cmd_params)))
        with sdk2.helpers.ProcessLog(self, logger='maps_edge_seq_stat_builder') as l:
            sdk_subprocess.check_call([executable] + cmd_params, stdout=l.stdout, stderr=l.stderr)

        logging.info('Success')

    def _get_last_table(self, path):
        import yt.wrapper as yt
        yt.config["proxy"]["url"] = self.Parameters.yt_proxy
        tables = list(yt.list(path))
        tables.sort(reverse=True)
        result = tables[0]
        logging.info('{} is last table in {}'.format(
            result,
            path))
        return result

    def _get_last_version(self):
        result = None
        try:
            lines = self.ecstatic(
                self.Parameters.ecstatic_environment,
                [
                    'versions',
                    self.Parameters.ecstatic_dataset,
                    self.Parameters.ecstatic_branch
                ],
                tvm_id=TVM_IDS.get(self.Parameters.ecstatic_environment),
                tvm_secret_id=TVM_SEC_UUIDS.get(self.Parameters.ecstatic_environment),
            ).splitlines()
        except subprocess.CalledProcessError as e:
            self.set_info(e.output)
            logging.error('Ecstatic returned {}: {}'.format(e.returncode, e.output))
            raise TaskFailure('Ecstatic returned ' + str(e.returncode))

        logging.info('Vesrsions exist in ecstatic:')
        for line in lines:
            logging.info(line)
            for dataset, sep, version in (line.strip().partition('='), ):
                if dataset == self.Parameters.ecstatic_dataset and sep == '=':
                    if (not result) or result < version:
                        result = version
        logging.info('Last version{} '.format(result))
        return result

    def _upload_ecstatic(self, dir_name, version):
        logging.info('Upload edge sequences statistics...')
        try:
            output = self.ecstatic(
                self.Parameters.ecstatic_environment,
                [
                    'upload',
                    self.Parameters.ecstatic_dataset + '=' + version,
                    dir_name,
                    '+' + self.Parameters.ecstatic_branch
                ],
                tvm_id=TVM_IDS.get(self.Parameters.ecstatic_environment),
                tvm_secret_id=TVM_SEC_UUIDS.get(self.Parameters.ecstatic_environment),
            )
        except subprocess.CalledProcessError as e:
            self.set_info(e.output)
            logging.error('Ecstatic returned {}: {}'.format(e.returncode, e.output))
            raise TaskFailure('Ecstatic returned ' + str(e.returncode))
        logging.info(output)
        logging.info('Success')

        if not self.Parameters.telegram_token:
            return

        telegram_token = sdk2.Vault.data(
            self.Parameters.telegram_token_owner,
            self.Parameters.telegram_token)
        bot = TelegramBot(telegram_token)
        bot.send_message(chat_id=self.Parameters.telegram_chat_id,
                         text="{}={} uploaded".format(self.Parameters.ecstatic_dataset, version))
