import logging
import os
import shutil
import subprocess

from datetime import datetime

from sandbox import sdk2
from sandbox.projects.maps.common.juggler_alerts import (
    TaskJugglerReportWithParameters
)
from sandbox.sandboxsdk import environments


class MapsCalcImpressionsCollectGeoproductInfoExecutable(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class MapsCalcImpressionsNmapsExecutable(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class MapsCalcImpressionsFetchFromRendererExecutable(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class MapsCalcImpressionsPostprocessRendererExecutable(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class MapsCalcImpressionsBebrExecutable(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class MapsCalcProtectedGeoproductImpressions(TaskJugglerReportWithParameters):
    # requrements for MULTISLOT tag
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 8 * 1024  # 8 GB
        environments = [
            environments.PipEnvironment('yql', use_wheel=True),
            environments.PipEnvironment('yandex-yt'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(TaskJugglerReportWithParameters.Parameters):

        owner = 'MAPS'
        kill_timeout = 24 * 60 * 60  # s

        with sdk2.parameters.Group('Task parameters') as main_parameters:
            yt_vault = sdk2.parameters.Vault('YT vault', required=True)
            yql_vault = sdk2.parameters.Vault('YQL vault', required=True)
            yt_pool = sdk2.parameters.String(
                'YT processing pool',
                default='maps-core-poi-impressions-checker',
                required=True
            )
            common_output_prefix = sdk2.parameters.String(
                'Common output prefix',
                required=True,
            )
            impressions_path = sdk2.parameters.String(
                'Impressions tables YT path',
                required=True,
            )

        with sdk2.parameters.Group('Collect info parameters') as collect_info_parameters:
            collect_info_executable = sdk2.parameters.Resource(
                'Sandbox resource ID of the executable to prepare geoproduct info table',
                resource_type=MapsCalcImpressionsCollectGeoproductInfoExecutable,
                required=True
            )

        with sdk2.parameters.Group('Nmaps parameters') as nmaps_parameters:
            nmaps_executable = sdk2.parameters.Resource(
                'Sandbox resource ID of the executable to calc nmaps impressions',
                resource_type=MapsCalcImpressionsNmapsExecutable,
                required=True
            )

        with sdk2.parameters.Group('Bebr parameters') as bebr_parameters:
            bebr_executable = sdk2.parameters.Resource(
                'Sandbox resource ID of the executable to calc bebr impressions',
                resource_type=MapsCalcImpressionsBebrExecutable,
                required=True,
            )

        with sdk2.parameters.Group('Renderer parameters') as renderer_parameters:
            self_tvm_id = sdk2.parameters.Integer(
                'Current TVM app id',
                default=2025240,
            )
            tvm_secret_vault = sdk2.parameters.Vault(
                'Current app TVM secret',
                required=True
            )
            renderer_host = sdk2.parameters.String(
                'Renderer host',
                required=True,
            )
            renderer_tvm_id = sdk2.parameters.Integer(
                'Renderer TVM app id',
                default=2023928,
            )
            fetch_from_renderer_threads_num = sdk2.parameters.Integer(
                'Threads amount for fetching from renderer',
                default=16,
            )

    def _load_executable(self, resource_id, name):
        resource = sdk2.Resource[resource_id]
        resource_data = sdk2.ResourceData(resource)
        executable_path = './{}'.format(name)
        shutil.copyfile(str(resource_data.path), executable_path)
        subprocess.check_call('chmod +x {}'.format(executable_path), shell=True)

    def _push_impressions_info(self, today, impressions_table, geoproduct_table, output_path):

        import yql.api.v1.client

        yql_token = self.Parameters.yql_vault.data()

        query = '''
            INSERT INTO `{output_path}` WITH TRUNCATE
            SELECT \n\tDISTINCT `l.permalink` as permalink, ''' + \
            ", ".join("\n\tNVL(`r.impressions_{zoom}`, 0) as impressions_{zoom}".format(zoom=zoom) for zoom in range(10, 22)) + \
            ", \n\t'{date}' as `date`, \n\tNVL(`r.is_indoor_covered`, false) as is_indoor_covered" + \
            '''
            FROM `{geoproduct_table}` as l
            LEFT JOIN `{impressions_table}` AS r
            ON l.permalink == r.permalink
        '''

        query = query.format(output_path=output_path, geoproduct_table=geoproduct_table, impressions_table=impressions_table, date=today)

        with yql.api.v1.client.YqlClient(db='hahn', token=yql_token) as yql_client:
            query = yql_client.query(query=query, syntax_version=1)
            query.run()

            results = query.get_results(wait=True)
            if not results.is_success:
                raise Exception('Unsuccessful request')

    def on_execute(self):
        import yt.wrapper as yt

        yt.config['proxy']['url'] = 'hahn'
        yt.config['token'] = self.Parameters.yt_vault.data()

        os.environ['YT_TOKEN'] = self.Parameters.yt_vault.data()
        os.environ['YQL_TOKEN'] = self.Parameters.yql_vault.data()
        os.environ['IMPRESSIONS_TVM_SECRET'] = self.Parameters.tvm_secret_vault.data()

        common_prefix = str(self.Parameters.common_output_prefix)
        today = datetime.strftime(datetime.today(), '%Y-%m-%d')
        impressions_path = str(self.Parameters.impressions_path)
        yt_pool = str(self.Parameters.yt_pool)

        collect_info_output_table = os.path.join(common_prefix, 'geoproduct_info', today)
        nmaps_output_table = os.path.join(common_prefix, 'nmaps', today)
        bebr_output_table = os.path.join(common_prefix, 'bebr', today)
        postprocess_renderer_output_table = os.path.join(common_prefix, 'renderer', today)
        impressions_table = os.path.join(impressions_path, today)

        self._load_executable(self.Parameters.collect_info_executable, 'collect_geoproduct_info')
        with self.memoize_stage.collect_geoproduct_info:
            cmd = [
                './collect_geoproduct_info',
                '--output-table',
                collect_info_output_table,
                '--date',
                today,
                '--yt-pool',
                yt_pool,
            ]
            logging.info('Running command %s', ' '.join(cmd))
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('collect info')) as pl:
                subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)

        self._load_executable(self.Parameters.nmaps_executable, 'nmaps')
        with self.memoize_stage.nmaps:
            cmd = [
                './nmaps',
                '--protected-geo-product-table',
                collect_info_output_table,
                '--output-table',
                nmaps_output_table,
                '--environment',
                'stable',
            ]
            logging.info('Running command %s', ' '.join(cmd))
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('nmaps')) as pl:
                subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)

        self._load_executable(self.Parameters.bebr_executable, 'bebr')
        with self.memoize_stage.bebr:
            cmd = [
                './bebr',
                '--output-table',
                bebr_output_table,
                '--yt-pool',
                yt_pool,
            ]
            logging.info('Running command %s', ' '.join(cmd))
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('bebr')) as pl:
                subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)

        self._push_impressions_info(today, impressions_table, collect_info_output_table, postprocess_renderer_output_table)

        links_list = [
            (collect_info_output_table, 'geoproduct_info'),
            (nmaps_output_table, 'nmaps'),
            (bebr_output_table, 'bebr'),
            (postprocess_renderer_output_table, 'renderer'),
        ]
        for table, link_name in links_list:
            yt.link(table, os.path.join(common_prefix, 'latest', link_name), force=True)
