import logging

from datetime import datetime, timedelta
from os import environ

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.sandboxsdk import environments


TTIMES_DATE_FORMAT = '%Y-%m-%d'
CLUSTER = 'hahn'
QUERY_TEMPLATE = '''
USE hahn;

INSERT INTO
    `{output_table}`
WITH TRUNCATE
SELECT
    persistent_id,
    COUNT(speed) as stats_speed_count,
    PERCENTILE(speed, 0.1) as stats_speed_p10,
    PERCENTILE(speed, 0.15) as stats_speed_p15,
    PERCENTILE(speed, 0.2) as stats_speed_p20,
    PERCENTILE(speed, 0.25) as stats_speed_p25,
    AVG(speed) as stats_speed_avg,
    PERCENTILE(speed, 0.55) / PERCENTILE(speed, 0.45) as stats_speed_p55_d_p45,
    PERCENTILE(speed, 0.6) / PERCENTILE(speed, 0.4) as stats_speed_p60_d_p40,
    PERCENTILE(speed, 0.7) / PERCENTILE(speed, 0.3) as stats_speed_p70_d_p30,
    PERCENTILE(speed, 0.75) / PERCENTILE(speed, 0.25) as stats_speed_p75_d_p25
FROM (
    SELECT
        persistent_id, (geolength / travel_time) * 3.6 as speed
    FROM RANGE(
        "//home/maps/jams/production/data/travel_times",
        "{start_date}",
        "{end_date}"
    )
)
GROUP BY persistent_id
HAVING COUNT(speed) >= 24 * {num_days};
'''


class MapsCalculateZeroSpeedBanerStats(sdk2.Task):
    """Calculate zero-speed-baner edge_stats table"""

    class Requirements(sdk2.Task.Requirements):
        environments = [environments.PipEnvironment('yql')]

    class Parameters(sdk2.Parameters):
        owner = 'MAPS'

        yt_token = sdk2.parameters.Vault('yt token vault', required=True)
        yql_token = sdk2.parameters.Vault('yql token vault', required=True)
        end_date = sdk2.parameters.String(
            'travel times end date',
            default=datetime.strftime(datetime.today(), TTIMES_DATE_FORMAT),
            required=True
        )
        num_days = sdk2.parameters.Integer(
            'num days to process before (including) the end date',
            default=14,
            required=True
        )
        output_table = sdk2.parameters.String('output yt table', required=True)

    def on_execute(self):
        from yql.api.v1.client import YqlClient

        environ['YQL_TOKEN'] = self.Parameters.yql_token.data()
        environ['YT_TOKEN'] = self.Parameters.yt_token.data()

        start_date = datetime.strptime(self.Parameters.end_date, TTIMES_DATE_FORMAT)
        start_date -= timedelta(days=self.Parameters.num_days - 1)
        start_date = datetime.strftime(start_date, TTIMES_DATE_FORMAT)

        query = QUERY_TEMPLATE.format(
            start_date=start_date,
            end_date=self.Parameters.end_date,
            num_days=self.Parameters.num_days,
            output_table=self.Parameters.output_table
        )
        yql_client = YqlClient(
            db=CLUSTER,
            token=self.Parameters.yql_token.data(),
        )
        logging.info('Running query:')
        logging.info(query)

        request = yql_client.query(query, syntax_version=1)
        request.run()
        request.get_results(wait=True)

        if not request.is_success:
            raise TaskError('YQL query failed')
