import json
import logging
import os
import re
import shlex

from sandbox.common.errors import SubprocessError
import sandbox.common.types.misc as ctm

from sandbox import sdk2

from sandbox.projects.common.build.YaPackage2 import YaPackage2
from sandbox.projects.common.build.ya_package_config.consts import PACKAGES_FILE

from sandbox.projects.maps.common import utils

logger = logging.getLogger(__name__)

DOCKER_CONFIG_FILE = '/etc/docker/daemon.json'

LXC_WITH_RESTARTABLE_DOCKER = 2619608374  # 1107712169

DEFAULT_TEST_SCRIPT_PATH = '/etc/runner/postbuild_test.sh'

MAPS_DOCKER_GROUP_NAME = 'Maps Docker parameters'

class RamdriveSize(sdk2.parameters.Integer):
    name = 'ramdrive_size'
    description = 'Create a RAM drive of specified size in Gb for dockerd'
    default_value = 30
    group = MAPS_DOCKER_GROUP_NAME


class TestScriptPath(sdk2.parameters.String):
    name = 'test_script_path'
    description = 'Script to be run after docker build'
    default_value = DEFAULT_TEST_SCRIPT_PATH
    group = MAPS_DOCKER_GROUP_NAME


class MapsDocker(YaPackage2):
    """
        Task for building and releasing docker images to "maps" Docker image repository.

        This task is a child of YA_PACKAGE_2 with some custom logic and custom default parameters.
    """

    class Parameters(YaPackage2.Parameters):
        ramdrive_size = RamdriveSize()
        test_script_path = TestScriptPath()

    def on_enqueue(self):
        if self.Parameters.ramdrive_size:
            MB_IN_GB = 1024
            self.Requirements.ramdrive = ctm.RamDrive(
                ctm.RamDriveType.TMPFS, int(self.Parameters.ramdrive_size) * MB_IN_GB, None)
        if not self.Parameters.sandbox_container:
            self.Parameters.sandbox_container = LXC_WITH_RESTARTABLE_DOCKER
        super(MapsDocker, self).on_enqueue()

    def on_execute(self):
        if self.ramdrive:
            self.patch_docker_to_use_ram()

        super(MapsDocker, self).on_execute()
        self.run_postbuild_docker_tests()
        self.set_ttl()

    def set_ttl(self):
        resources = self.Context.package_resources['resources']
        assert len(resources) == 1
        resource_id, _ = resources[0]  # ignore debug resource
        build_resource = sdk2.Resource[resource_id]
        build_resource.ttl = 365 * 2  # ~2 years
        build_resource.backup_task = True

    def patch_docker_to_use_ram(self):
        with open(DOCKER_CONFIG_FILE, 'r') as file:
            docker_config = json.load(file)
        docker_config['data-root'] = os.path.join(str(self.ramdrive.path), 'docker')
        with open(DOCKER_CONFIG_FILE, 'w') as file:
            json.dump(docker_config, file, indent=4)
        logger.info('docker config at {} edited to:\n{}'.format(DOCKER_CONFIG_FILE, json.dumps(docker_config)))

        process = self._subprocess('sudo service docker restart', wait=True)
        if process.returncode != 0:
            raise Exception('Failed to restart docker')
        logger.info('docker restarted')

        self._subprocess('docker info', wait=True)

    def prebuild_hook(self):
        # Extract packages names from pkg.json and mark corresponding resources with this name
        self.mark_resources()
        # Pull parent docker image in parallel with ya packaging
        self.preload_parent_image()

    def mark_resources(self):
        package_resources = self.Context.package_resources
        packages, resources = package_resources['packages'], package_resources['resources']
        vcs_info = utils.vcs_info(
            arcadia_url=self.Parameters.checkout_arcadia_from_url,
            arcadia_mount_path=self.arcadia_src_dir,
        )
        for package, (resource_id, debug_resource_id) in zip(packages, resources):
            with open(os.path.join(self.arcadia_src_dir, package), 'r') as file:
                pkg_json = json.load(file)
            resource = sdk2.Resource[resource_id]
            resource.resource_name = pkg_json['meta']['name']
            resource.svn_revision = vcs_info['svn_revision_ya_package_compatible']
            resource.arc_commit = vcs_info['arc_commit_hash'] or ''
            resource.branch = vcs_info['branch'] or ''

    def preload_parent_image(self):
        packages = self.Context.package_resources['packages']
        for package in packages:
            logger.info('looking for Dockerfile in {}'.format(package))
            with open(os.path.join(self.arcadia_src_dir, package), 'r') as file:
                pkg_json = json.load(file)

            dockerfile_path = self._deduce_dockerfile_path(pkg_json, os.path.dirname(package))
            if not dockerfile_path:
                continue
            logger.info('Dockerfile found at {}'.format(dockerfile_path))
            with open(os.path.join(self.arcadia_src_dir, dockerfile_path), 'r') as file:
                dockerfile_text = file.read()

            parent_image = self._deduce_parent_image(dockerfile_text)
            if not parent_image:
                continue
            logger.info('parent image {} found'.format(parent_image))
            self.Context.parent_docker_image = parent_image
            self._docker_login()
            self._subprocess('docker pull {}'.format(parent_image), wait=False)

    @classmethod
    def _deduce_dockerfile_path(cls, pkg_json, pkg_path):
        for item in pkg_json.get('data', []):
            destination_path = item.get('destination', {}).get('path')
            if destination_path and os.path.normpath(destination_path) == '/Dockerfile':
                source_path = item.get('source', {}).get('path')
                if item.get('source', {}).get('type') == 'ARCADIA':
                    return source_path
                elif item.get('source', {}).get('type') == 'RELATIVE':
                    return os.path.join(pkg_path, source_path)

    @classmethod
    def _deduce_parent_image(cls, dockerfile_text):
        match = re.search(r'^[ \t]*FROM[ \t]+(registry.yandex.net/[\w./-]+:[\w.-]+)[ \t]*$', dockerfile_text, re.MULTILINE)
        if match:
            return match.group(1)

    def run_postbuild_docker_tests(self):
        test_script_path = self.Parameters.test_script_path
        if not test_script_path:
            return
        if not os.path.exists(PACKAGES_FILE):
            error_message = 'Cannot find {} output file from ya package'.format(PACKAGES_FILE)
            logger.error(error_message)
            self.set_info(error_message)
            return
        with open(PACKAGES_FILE) as packages_file:
            for package in json.load(packages_file):
                if "docker_image" in package:
                    self.run_docker_test(package["docker_image"], test_script_path)

    def run_docker_test(self, docker_image, test_script_path):
        FAILED_TAG = 'POSTBUILD_TEST_FAIL'
        LOG_PREFIX = 'postbuild_test'
        logger.info('Starting test for {}'.format(docker_image))
        process = self._subprocess(
            "docker run {docker_image} bash -c 'if [ -f {test_script_path} ]; then {test_script_path}; fi'".format(
                docker_image=docker_image,
                test_script_path=test_script_path
            ),
            wait=True,
            check=False,
            log_prefix=LOG_PREFIX
        )
        if process.returncode != 0:
            self.Parameters.tags += [FAILED_TAG]
            error_message = 'Test of {} failed, check {}.err and {}.out for details'.format(
                docker_image, LOG_PREFIX, LOG_PREFIX)
            self.set_info(error_message)
            raise Exception(error_message)

    def _subprocess(self, args, wait=False, check=True,
                    log_prefix=None, exception_type=SubprocessError):
        if isinstance(args, str):
            args = shlex.split(args)
        assert len(args) >= 1
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(log_prefix or args[0])) as pl:
            process = sdk2.helpers.subprocess.Popen(args, stdout=pl.stdout, stderr=pl.stderr)
            if wait:
                process.wait()
            if wait and check and process.returncode:
                raise exception_type('process {} died with exit code {}'
                                     .format(args, process.returncode))
            return process
