# -*- coding: utf-8 -*-

import logging
import logging.handlers
import os
import re
import time

from sandbox.projects.common import file_utils as fu

from sandbox import sdk2
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt

SHOTS_DIR = 'shots'


def logger():
    loggerr = logging.getLogger('%s_%s' % (__name__, time.time()))
    loggerr.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s %(levelname)s [%(processName)s: %(threadName)s] %(message)s')
    file_handler = logging.handlers.RotatingFileHandler(
        'maps_matrix_router_shooting.log',
        maxBytes=1024 * 1024,
        backupCount=5
    )

    file_handler.setLevel(logging.DEBUG)
    file_handler.setFormatter(formatter)
    loggerr.addHandler(file_handler)
    return loggerr


class DrivingMatrixRouterShooting(sdk2.Task):
    """ Task for shooting at matrix_router st/MAPSCORE-4875 """
    local_configs = ''

    class Parameters(sdk2.Parameters):
        st_ticket = sdk2.parameters.String('ST/Ticket', required=True)
        nanny_load_service = sdk2.parameters.String('Nanny service name to shoot on', required=True)
        arcadia_regression_path = sdk2.parameters.String('Path to regression configs from arcadia root', required=True)

        with sdk2.parameters.Output:
            lunapark_links = sdk2.parameters.List('Lunapark links', required=True)

    def start_shooting(self, desc, ammo_source, ammo_resource, config_content):
        logger().info('Nanny service name to shoot on: %s', self.Parameters.nanny_load_service)
        logger().info('Path to regression configs from arcadia trunk root: %s', self.Parameters.arcadia_regression_path)
        monitoring_path = os.path.join(self.Parameters.arcadia_regression_path, 'telegraf.xml')
        subtask_shoot = ShootViaTankapi(
            self,
            description=desc,
            nanny_service=self.Parameters.nanny_load_service,
            ammo_source=ammo_source,
            ammo_resource=ammo_resource,
            config_source='file',
            config_content=config_content,
            use_monitoring=True,
            monitoring_source="arcadia",
            monitoring_arc_path=monitoring_path,
            config_add_parameters=' -o uploader.task={}'.format(self.Parameters.st_ticket),
            use_public_tanks=True
        ).enqueue()
        logger().info('Subtask with shooting is started')
        raise sdk2.WaitTask([subtask_shoot.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True,
                            timeout=14400)

    def on_prepare(self):
        os.makedirs('configs', 0o755)
        self.local_configs = os.path.join(str(self.path('configs')), SHOTS_DIR)
        tank_config_dir = sdk2.svn.Arcadia.trunk_url(os.path.join(self.Parameters.arcadia_regression_path, 'configs'))
        sdk2.svn.Arcadia.export(tank_config_dir, self.local_configs)

    def on_execute(self):
        tank_configs = os.listdir(self.local_configs)
        logger().info('Tank configs: %s', tank_configs)

        for conf_name in tank_configs:
            with self.memoize_stage["shooting_{0}".format(conf_name)]:
                config_content = fu.read_file(os.path.join(self.local_configs, conf_name))
                self.start_shooting(conf_name, 'in_config', '', config_content)

        self.Parameters.lunapark_links = [
            shoot.Parameters.lunapark_link for shoot in self.find(ShootViaTankapi)]
