from sandbox import common
import subprocess
import re
import os.path

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.telegram import TelegramBot

from sandbox.projects.maps.common.ecstatic_bin import MapsEcstaticToolMixin


ECSTATIC_AUTH_CREDENTIALS = {
    'stable': {
        'tvm_id': 2019970,
        'tvm_secret_id': 'sec-01e6xt53a0pcx62f0vf3zt54y7'
    },
    'testing': {
        'tvm_id': 2019972,
        'tvm_secret_id': 'sec-01e6xt5e73ggr4te2ah5x5521g'
    },
    'datatesting': {
        'tvm_id': 2023440,
        'tvm_secret_id': 'sec-01ej69gcrzs99rsr0q91p05w2v'
    },
    'unstable': {
        'tvm_id': 2019974,
        'tvm_secret_id': 'sec-01e6xt5vvr5bzkntgj01mk3f5c'
    }
}

class MapsGeodataUploader(MapsEcstaticToolMixin, sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):

        geodata_resource_type = sdk2.parameters.String(
            'Geodata resource type to upload', default='GEODATA4BIN_STABLE')
        untar_archive = sdk2.parameters.Bool('Needs to untar?', default=True)
        dataset_name = sdk2.parameters.String('Dataset name', default='geodata4', required=True)
        environments = sdk2.parameters.List('Environments', default=['testing'], required=True)
        branches = sdk2.parameters.List('Branches', default=['testing', 'stable'], required=True)

        geodata_resource = sdk2.parameters.Resource('Geodata resource id (optional)')
        version = sdk2.parameters.String('Dataset version (optional)')
        debug = sdk2.parameters.Bool('Debug ymtorrent', default=False)

        telegram_notification = sdk2.parameters.Bool(
            'Notify telegram chat?',
            default=False
        )
        with telegram_notification.value[True]:
            with sdk2.parameters.Group('Telegram notification parameters') as telegram_parameters:
                telegram_token = sdk2.parameters.String(
                    'Telegram token sandbox vault record name'
                )
                telegram_token_owner = sdk2.parameters.String(
                    'Telegram token sandbox vault record owner'
                )
                telegram_chat_id = sdk2.parameters.Integer(
                    'Chat id', default=-1001465293474
                )

    def on_execute(self):
        def latest_geobase_resource(resource_type):
            return sdk2.Resource[resource_type].find(owner='GEOBASE').order(-sdk2.Resource.id).first()

        resource = self.Parameters.geodata_resource or (
            latest_geobase_resource(self.Parameters.geodata_resource_type))
        self.set_info('Using data from: ' + str(resource))

        dataset = self.Parameters.dataset_name
        version = self.Parameters.version or re.sub(r'\W', '-', str(resource.created))
        branches = ['+' + branch for branch in self.Parameters.branches]
        self.debug_ymtorrent = self.Parameters.debug

        resource_path = unicode(sdk2.ResourceData(resource).path)
        if self.Parameters.untar_archive:
            data_dir = unicode(self.path('dataset_dir'))
            common.fs.make_folder(data_dir)
            common.fs.untar_archive(resource_path, data_dir)
        else:
            data_dir = os.path.dirname(resource_path)

        try:
            for environment in self.Parameters.environments:
                self.set_info(
                    'Uploading {dataset}={version} to {environment} and moving {branches_str}'.format(
                    branches_str=', '.join(branches), **locals()
                ))
                self.ecstatic(
                    environment=environment,
                    do_auth=True,
                    cmd_args=['upload', dataset + '=' + version, data_dir] + branches, **ECSTATIC_AUTH_CREDENTIALS[environment])
        except subprocess.CalledProcessError as e:
            self.set_info(e.output)
            raise TaskFailure('Ecstatic returned ' + str(e.returncode))

        if self.Parameters.telegram_notification:
            telegram_token = sdk2.Vault.data(
                self.Parameters.telegram_token_owner,
                self.Parameters.telegram_token)
            bot = TelegramBot(telegram_token)
            bot.send_message(chat_id=self.Parameters.telegram_chat_id,
                             text='{}={} uploaded'.format(dataset, version))

