# -*- coding: utf-8 -*-

import logging
import logging.handlers
import os
import time
import urllib2
import yaml

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import task as ctt
from sandbox.projects.common import file_utils as fu
from sandbox.projects.maps.MapsGraphTileBuilderAmmoGen import MapsGraphTileBuilderAmmoGen
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from sandbox.projects.tank.ShootViaTankapi.detect_tank import TankFinder

ARCADIA_REGRESSION_PATH = 'maps/routing/graph-tile-builder/regression'
GRAPH_VERSION_URL = 'http://{0}/graph_version'
NANNY_LOAD_SERVICE = 'maps_core_graph_tile_builder_load'


def logger():
    loggerr = logging.getLogger('%s_%s' % (__name__, time.time()))
    loggerr.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s %(levelname)s [%(processName)s: %(threadName)s] %(message)s')
    file_handler = logging.handlers.RotatingFileHandler(
        'maps_routing_shooting.log',
        maxBytes=1024 * 1024,
        backupCount=5
    )

    file_handler.setLevel(logging.DEBUG)
    file_handler.setFormatter(formatter)
    loggerr.addHandler(file_handler)
    return loggerr


class MapsGraphTileBuilderShooting(sdk2.Task):
    """ Task for shooting on st/MAPSCORE-4681 """
    local_configs = ''

    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 2048  # 2GB

    class Parameters(sdk2.Parameters):
        stTicket = sdk2.parameters.String('ST/Ticket', required=True)
        with sdk2.parameters.Output:
            lunapark_links = sdk2.parameters.List('Lunapark links', required=True)

    def start_shooting(self, desc, ammo_source, ammo_resource, config_content):
        monitoring_path = os.path.join(ARCADIA_REGRESSION_PATH, 'telegraf.xml')
        subtask_shoot = ShootViaTankapi(
            self,
            description=desc,
            ammo_source=ammo_source,
            ammo_resource=ammo_resource,
            config_source='file',
            config_content=config_content,
            use_monitoring=True,
            monitoring_source="arcadia",
            monitoring_arc_path=monitoring_path,
            nanny_service=NANNY_LOAD_SERVICE,
            config_add_parameters=' -o uploader.task={}'
                .format(self.Parameters.stTicket),
            use_public_tanks=True
        ).enqueue()
        logger().info('Subtask with shooting is started')
        raise sdk2.WaitTask([subtask_shoot.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True,
                            timeout=14400)

    def on_prepare(self):
        os.makedirs('configs', 0o755)
        self.local_configs = os.path.join(str(self.path('configs')), 'shoots')
        tank_config_dir = sdk2.svn.Arcadia.trunk_url(os.path.join(ARCADIA_REGRESSION_PATH, 'configs'))
        sdk2.svn.Arcadia.export(tank_config_dir, self.local_configs)

    def detect_target_host(self, nanny_service):
        tank_finder = TankFinder(target_nanny_service=nanny_service)
        target_hosts = tank_finder.define_hosts_in_nanny_group(group_name=nanny_service)
        if len(target_hosts) > 0:
            logger().info('Detected hostname %s in nanny service %s.', target_hosts[0].host, nanny_service)
            return target_hosts[0].host
        else:
            logger().info('Can not detect hostname in nanny service %s.', nanny_service)
            raise errors.TaskError('No target host was found!')

    def on_execute(self):
        tank_configs = os.listdir(self.local_configs)
        logger().info('Tank configs: %s', tank_configs)

        graph_version_url = GRAPH_VERSION_URL.format(self.detect_target_host(NANNY_LOAD_SERVICE))
        logger().info('graph version url = %s', graph_version_url)
        for config_name in tank_configs:
            with open(os.path.join(self.local_configs, config_name), 'r') as f:
                yaml_content = yaml.load(f)
            yaml_ammo_url = yaml_content['phantom']['ammofile']
            config_content = fu.read_file(os.path.join(self.local_configs, config_name))
            with self.memoize_stage['ammo_generating_{0}'.format(config_name)]:
                graph_version = urllib2.urlopen(graph_version_url).read()
                logger().info('graph version = %s', graph_version)
                ammo_generator_task = MapsGraphTileBuilderAmmoGen(
                    self,
                    description='ammo generator for task {0} and config_name[{1}]'.
                        format(self.id, config_name),
                    ammo_url=yaml_ammo_url,
                    required_graph_version=graph_version).enqueue()
                self.Context.ammo_generator_task_id = ammo_generator_task.id
                self.Context.save()
                logger().info('ammo_generator_task_id = %s', ammo_generator_task.id)
                logger().info('Starting subtask: ' + ammo_generator_task.Parameters.description)
                raise sdk2.WaitOutput({ammo_generator_task.id: 'ammo'}, wait_all=True, timeout=3600)
            with self.memoize_stage['shooting_{0}'.format(config_name)]:
                ammo_generator_task = self.find(
                    MapsGraphTileBuilderAmmoGen,
                    id=self.Context.ammo_generator_task_id).first()
                logger().info('found ammo_generator_task_id = %s', ammo_generator_task.id)
                self.start_shooting(config_name, 'resource', ammo_generator_task.Parameters.ammo, config_content)

        self.Parameters.lunapark_links = [
            shoot.Parameters.lunapark_link for shoot in self.find(ShootViaTankapi)]
