import os
import logging
import requests

import sandbox.common.types.task as task

from sandbox import sdk2
from sandbox.projects.tank.load_resources.resources import YANDEX_TANKAPI_LXC_CONTAINER
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from sandbox.common.types import resource as resource

NANNY_SERVICE = 'maps_core_jams_infopoints_load'


def create_local_dir(dir):
    try:
        os.makedirs(dir, 0o755)
    except OSError:
        pass


class MapsInfopointShooting(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 5 * 1024  # 5GB

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        target = sdk2.parameters.String(
            'infopoint backend (fqdn)', required=True)
        configs = sdk2.parameters.List(
            'shooting configs',
            required=True,
            description=("Shooting configs list. "
                         "Value is yaml config arc path (e.g. maps/...)"))
        with sdk2.parameters.Output:
            lunapark_links = sdk2.parameters.List(
                'Lunapark links', required=True)

    def _notifications_only_on_failures(self):
        notifications = []
        for notification in self.Parameters.notifications:
            notifications.append(sdk2.Notification(
                statuses=[task.Status.FAILURE, task.Status.EXCEPTION, task.Status.EXPIRED, task.Status.TIMEOUT],
                recipients=notification.recipients,
                transport=notification.transport))
        return notifications

    def _enqueue_shooting(self, target, config_arc_path, tag):
        description = 'Shooting stage {}. Target: {}. Config: {}'.format(
            tag, target, config_arc_path)
        logging.info(description)
        container = YANDEX_TANKAPI_LXC_CONTAINER.find(
            state=resource.State.READY).order(-YANDEX_TANKAPI_LXC_CONTAINER.id).first().id
        shooting_task = ShootViaTankapi(
            self,
            description=description,
            use_public_tanks=True,
            nanny_service=NANNY_SERVICE,
            notifications=self._notifications_only_on_failures(),
            config_add_parameters="-o phantom.address={}".format(target),
            config_source='arcadia',
            config_arc_path=config_arc_path,
            use_monitoring=True,
            monitoring_source='in_config',
            ammo_source='in_config',
            container=container
        )
        # https://wiki.yandex-team.ru/maps/dev/core/instrukcija-po-sozdaniju-servisov-v-rtc/#semaforyeslivyzapuskaeteshootviatankapivkachestvepodzadachi
        shooting_task.Requirements.semaphores = task.Semaphores(
            acquires=[
                task.Semaphores.Acquire(name='SAS_MAPS_CORE_SHOOT_VIA_TANKAPI')
            ],
            release=(
                task.Status.Group.BREAK,
                task.Status.Group.FINISH,
                task.Status.Group.WAIT
            )
        )
        shooting_task.save().enqueue()
        self.set_info('Starting subtask: ' + description)
        return shooting_task

    def _clean(self, target, tag):
        description = 'Cleaning stage {}. Target: {}'.format(
            tag, target)
        self.set_info('Starting subtask: ' + description)

        resp = requests.get(
            "http://{}/wipe_data".format(target),
            headers={"Host": "infopoints.load.maps.yandex.net"})

        self.set_info(
            'Finished cleaning: {} {}'.format(resp.status_code, resp.text))

    def on_execute(self):
        for index, config_arc_path in enumerate(self.Parameters.configs):
            # Shooting stage
            with self.memoize_stage['shooting_stage_{}'.format(index)]:
                shooting = self._enqueue_shooting(
                    self.Parameters.target, config_arc_path, index)

                raise sdk2.WaitTask([shooting.id],
                                    task.Status.Group.FINISH | task.Status.Group.BREAK,
                                    wait_all=True,
                                    timeout=3600)
            # Cleaning stage
            with self.memoize_stage['cleaning_stage_{}'.format(index)]:
                self._clean(self.Parameters.target, index)

        self.Parameters.lunapark_links = [
            shoot.Parameters.lunapark_link for shoot in self.find(ShootViaTankapi)]
