# -*- coding: utf-8 -*-

import logging
import time
import uuid
import urllib2

from sandbox import sdk2
from sandbox.projects.common import file_utils

from sandbox.projects.maps.MapsInfopointsDangerUpdate.utils.template import Template

UPLOAD_TIMEOUT = 60 * 20  # 20 minutes
UUID_NAMESPACE = uuid.UUID("98f60536-f70b-4f95-be82-0b92fbf8ed59")


class InfopointsDangerXmlResource(sdk2.Resource):
    # Generated xml
    releasable = True
    partner = sdk2.parameters.String


class MapsInfopointsDangerUpdate(sdk2.Task):
    class Parameters(sdk2.Parameters):
        partner = sdk2.parameters.String("Partner")
        with sdk2.parameters.Group("Resource") as arcadia_parameters:
            raw_points = sdk2.parameters.Resource("Raw points resource")

        with sdk2.parameters.Group("Testing deployment") as deploy_test:
            test_aggregator_url = sdk2.parameters.String("Aggregator url to send sync request", required=True)

        with sdk2.parameters.Group("Production deployment") as deploy_prod:
            prod_aggregator_url = sdk2.parameters.String("Aggregator url to send sync request")

    def on_execute(self):
        logging.info("Executing...")

        self.prepare_events()
        self.generate_xml()
        self.sync_infopoints(self.Parameters.test_aggregator_url)

        logging.info("Executed successfully")

    def on_release(self, parameters_):
        logging.info("Releasing...")

        self.mark_released_resources("stable", ttl="inf")
        self.sync_infopoints(self.Parameters.prod_aggregator_url)

        super(MapsInfopointsDangerUpdate, self).on_release(parameters_)

        logging.info("Released successfully")

    def prepare_events(self):
        logging.info("Preparing events...")

        raw_points = sdk2.ResourceData(self.Parameters.raw_points)

        self.prepared_events = []

        for line in file_utils.read_lines(raw_points.path):
            if len(line.strip()) == 0:
                break

            lat, lon = map(str.strip, line.split(","))
            pos = " ".join([lon, lat])

            self.prepared_events.append({
                "type": "danger",
                "event_id": str(uuid.uuid5(UUID_NAMESPACE, pos)),
                "pos": pos,
                "description": "Аварийно-опасный участок"
            })

        logging.info("Prepared {} events".format(len(self.prepared_events)))

    def generate_xml(self):
        logging.info("Generating xml resource...")

        xml_resource = sdk2.ResourceData(InfopointsDangerXmlResource(
            self, self.Parameters.description, "infopoints.xml",
            partner=self.Parameters.partner
        ))

        xml_path = xml_resource.path

        start_time = int(time.time())
        end_time = start_time + 365 * 24 * 60 * 60  # + 1 year

        template = Template.get()

        file_utils.write_file(xml_path, template.render(
            events=self.prepared_events,
            start_time=start_time,
            end_time=end_time
        ))

        xml_resource.ready()

        logging.info("Xml resource successfully generated, saved to {}".format(xml_path))

    def sync_infopoints(self, aggregator_url):
        logging.info("Requesting infopoint aggregator to sync infopoints: {}".format(aggregator_url))

        request = urllib2.Request(aggregator_url)
        request.add_header("Host", "infopoints.maps.yandex.net")
        urllib2.urlopen(request, timeout=UPLOAD_TIMEOUT).read()

        logging.info("Successfully requested to sync infopoints")
