import logging
import os
import subprocess
import tarfile

from datetime import datetime

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import task
from sandbox.sandboxsdk import environments
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.common.constants import constants as sdk_constants
from sandbox.projects.common.vcs import aapi
from sandbox.projects.maps.common import utils
from sandbox.projects.maps.common.ecstatic_bin import MapsEcstaticToolMixin
from sandbox.projects.maps.common.mapscore_resources import MapsCoreCoverageResource
from sandbox.projects.maps.common.retry import retry

GEOID_NAME = 'geoid.mms.1'
DATASET_NAME = 'yandex-maps-coverage5-trf'
DATASET_DIR = 'dataset/'

JAMS_COVERAGE_ARCADIA_PATH = 'maps/jams/coverage/data'
JAMS_COVERAGE_TEST_ARCADIA_PATH = 'maps/jams/coverage/utils'

ECSTATIC_TESTING_ENV = 'testing'
ECSTATIC_STABLE_ENV = 'stable'
ECSTATIC_TESTING_BRANCH = 'testing'
ECSTATIC_STABLE_BRANCH = 'stable'

TVM_ID = 2017289
TVM_SECRET_ID = 'sec-01dvdpewvkke97hhz8sg360kya'


class MapsJamsTrfBuilder(sdk2.Task, MapsEcstaticToolMixin):
    """
    Task for building TRF coverage data
    """

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.SvnEnvironment(),
        )

    class Context(sdk2.Task.Context):
        version = datetime.now().strftime("%Y.%m.%d.%H%M%S")
        dataset_path = os.path.join(JAMS_COVERAGE_ARCADIA_PATH, DATASET_DIR)

    def on_prepare(self):
        self.arcadia_revision = aapi.ArcadiaApi.svn_head()
        self.resource = MapsCoreCoverageResource(
            self,
            path='trf.tar.gz',
            sandbox_task_id=sdk2.Task.current.id,
            sandbox_task_type=sdk2.Task.current.type.__name__,
            sandbox_task_owner=sdk2.Task.current.owner,
            arcadia_revision=self.arcadia_revision,
            version=self.Context.version,
            description='Maps coverage TRF dataset'
        )

    def on_execute(self):
        self._build_dataset()
        self._test_dataset()

        self._upload_archived_resource(self.resource, self.Context.dataset_path)
        self._upload_dataset(
            ECSTATIC_TESTING_ENV, DATASET_NAME, self.Context.version,
            self.Context.dataset_path, [ECSTATIC_TESTING_BRANCH, ECSTATIC_STABLE_BRANCH])
        self._upload_dataset(
            ECSTATIC_STABLE_ENV, DATASET_NAME, self.Context.version, self.Context.dataset_path, [])

    def on_release(self, params):
        logging.info("Releasing TRF dataset...")

        if (params['release_status'] == task.ReleaseStatus.TESTING):
            self._move_dataset(ECSTATIC_STABLE_ENV, DATASET_NAME, self.Context.version, [ECSTATIC_TESTING_BRANCH])
        elif (params['release_status'] == task.ReleaseStatus.STABLE):
            self._move_dataset(ECSTATIC_STABLE_ENV, DATASET_NAME, self.Context.version, [ECSTATIC_STABLE_BRANCH])
        else:
            raise errors.TaskFailure(
                "Unexpected relese branch: {}".format(params['release_status']))
        self.mark_released_resources(params['release_status'])

    def _build_dataset(self):
        logging.info("Building TRF dataset from revision {} ...".format(self.arcadia_revision))
        arcadia_url = sdk2.svn.Arcadia.ARCADIA_TRUNK_URL + "@{}".format(self.arcadia_revision)

        with arcadia_sdk.mount_arc_path(arcadia_url) as revision_root:
            arcadia_sdk.do_build(
                build_system=sdk_constants.YMAKE_BUILD_SYSTEM,
                source_root=revision_root,
                targets=[JAMS_COVERAGE_ARCADIA_PATH, JAMS_COVERAGE_TEST_ARCADIA_PATH],
                results_dir='./',
                clear_build=False,
            )

        utils.write_build_info_file(
            self.Context.dataset_path,
            self.arcadia_revision,
            self.Context.version,
            {"layer": "trf"})

    def _test_dataset(self):
        logging.info("Testing TRF dataset...")

        test_path = os.path.join(JAMS_COVERAGE_TEST_ARCADIA_PATH, 'trf_checker')
        geoid_path = os.path.join(JAMS_COVERAGE_ARCADIA_PATH, 'temp', GEOID_NAME)
        trf_path = os.path.join(JAMS_COVERAGE_ARCADIA_PATH, 'dataset', 'trf.mms.1')

        command = "{} -- -g {} -t {}".format(test_path, geoid_path, trf_path)
        res = utils.execute(self, command, "coverage.building")

        if (res != 0):
            raise errors.TaskFailure("TRF check failed: {}".format(res))

    def _upload_archived_resource(self, resource, data_path):
        logging.info("Uploading dataset to sandbox...")

        resource_data = sdk2.ResourceData(resource)
        with tarfile.open(str(resource_data.path), 'w:gz') as tar:
            for entry in os.listdir(data_path):
                tar.add(os.path.join(data_path, entry), arcname=entry)
        resource_data.ready()

    def _upload_dataset(self, env, dataset, version, dir, branches):
        logging.info(
            "Uploading {} dataset, version = {} to {}:{}".format(dataset, version, env, branches))
        branches = ['+' + branch for branch in branches]
        self._ecstatic_command(env, ['upload', dataset + '=' + version, dir] + branches)

    def _move_dataset(self, env, dataset, version, branches):
        logging.info(
            "Moving {} dataset, version = {} to {}:{}".format(dataset, version, env, branches))
        branches = ['+' + branch for branch in branches]
        self._ecstatic_command(env, ['move', dataset + '=' + version] + branches)

    def _ecstatic_command(self, env, params):
        try:
            return self._retry_call(lambda *args: self.ecstatic(*args, tvm_id=TVM_ID, tvm_secret_id=TVM_SECRET_ID))(env, params)
        except subprocess.CalledProcessError as ex:
            self.set_info(ex.output)
            logging.error('Ecstatic returned {}: {}'.format(
                ex.returncode, ex.output))
            raise errors.TaskFailure(
                "Couldn't execute ecstatic command, code = {}".format(ex.returncode))

    def _retry_call(self, *args):
        return retry(
            exceptions=subprocess.CalledProcessError,
            tries=5,
            delay=30,
            max_delay=120,
            backoff=2)(*args)

    @property
    def release_template(self):
        return sdk2.ReleaseTemplate(
            types=[
                task.ReleaseStatus.STABLE,
                task.ReleaseStatus.TESTING
            ]
        )
