# -*- coding: utf-8 -*-

import binascii
import logging
import os
import subprocess

import requests

import sandbox.common.errors
import sandbox.common.platform
import sandbox.common.types.resource
import sandbox.common.urls
import sandbox.sdk2

_JUGGLER_SERVICE_NAME = "shooting_failed"

_TVMTOOL_PORT = 12345
_TVMTOOL_CACHE_DIR = "tvmtool-cache-dir"


def _tvm_config(shooter_tvm_id, shooter_tvm_secret, router_tvm_id):
    return """{{
    "clients": {{
        "shooter": {{
            "secret": "{shooter_tvm_secret}",
            "self_tvm_id": {shooter_tvm_id},
            "dsts": {{
                "router": {{
                    "dst_id": {router_tvm_id}
                }},
                "mds-stable": {{
                    "dst_id": 2000273
                }}
            }}
        }}
    }}
}}
""".format(
        shooter_tvm_id=shooter_tvm_id,
        shooter_tvm_secret=shooter_tvm_secret,
        router_tvm_id=router_tvm_id)


def _random_auth_token():
    """Generate a random string to be used as a TVM authorization token.

    More on this token:
    https://wiki.yandex-team.ru/passport/tvm2/tvm-daemon/#kakzapustitprocess
    """
    return binascii.hexlify(os.urandom(16)).decode("utf-8")


def _prepare_tvmtool():
    platform = sandbox.common.platform.platform()
    arch = sandbox.common.platform.get_arch_from_platform(platform)
    tvmtool_resource = sandbox.sdk2.Resource.find(
        type="TVM_TOOL_BINARY", arch=arch).first()
    return str(sandbox.sdk2.ResourceData(tvmtool_resource).path)


class MatrixRouterShootingTask(sandbox.sdk2.Task):
    """Create load on matrix router in testing environment.

    https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/maps/MapsMatrixRouterShootingTask
    """

    # TODO(immartynov): Use one of the libraries:
    #   * sandbox/projects/common/juggler or
    #   * sandbox/projects/maps/common/juggler_alerts.py
    def _send_status_to_juggler(self, status, current_task_state):
        task_url = sandbox.common.urls.get_task_link(self.id)
        event = {
            "host": self.Parameters.juggler_host_name,
            "service": _JUGGLER_SERVICE_NAME,
            "status": status,
            "description": (
                "MatrixRouterShootingTask is in state {}: {}".format(
                    current_task_state, task_url)
            )
        }

        try:
            reply = requests.post(
                "http://juggler-push.search.yandex.net/events",
                json={"source": "sandbox", "events": [event]}
            )
            event_status = reply.json()["events"][0]
            if event_status["code"] != 200:
                raise Exception(event_status["error"])
        except Exception as e:
            logging.exception("could not send event to Juggler: {}".format(e))

    class Requirements(sandbox.sdk2.Requirements):
        cores = 1
        disk_space = 1024  # MB
        ram = 150 * 1024  # MB

    class Parameters(sandbox.sdk2.Task.Parameters):
        kill_timeout = 15 * 60  # seconds

        router_url = sandbox.sdk2.parameters.String(
            "URL of the matrix router to shoot, with 'http[s]://' prefix")
        router_tvm_id = sandbox.sdk2.parameters.Integer(
            "ID of the target router TVM application")
        shooter_tvm_id = sandbox.sdk2.parameters.Integer(
            "ID of the shooter TVM application")
        shooter_tvm_secret = sandbox.sdk2.parameters.YavSecret(
            "ID of the yav secret created for the shooter TVM application")
        juggler_host_name = sandbox.sdk2.parameters.String(
            "Host of the juggler event used to report the status")

    def _tvm_secret(self):
        return self.Parameters.shooter_tvm_secret.data()["client_secret"]

    def on_create(self):
        if self.Requirements.tasks_resource is None:
            # This code does not seem to work when running a task locally with
            # --enable-taskbox. The if-else solves this.
            logging.info("task binary is not set, looking for latest suitable one")
            self.Requirements.tasks_resource = (
                sandbox.sdk2.service_resources.SandboxTasksBinary.find(
                    attrs={"target": "maps/MatrixRouterShootingTask"},
                    state=sandbox.common.types.resource.State.READY
                ).first()
            )
            logging.info("task binary: {}".format(self.Requirements.tasks_resource))
        else:
            logging.info("task binary is already set")

    def on_execute(self):
        logging.info("checking that tasks resource is properly set")
        logging.info("current tasks_resource: {}".format(
            self.Requirements.tasks_resource))
        if self.Requirements.tasks_resource is None:
            raise sandbox.common.errors.TaskFailure(
                "self.Requirements.tasks_resource is not set"
                " for MatrixRouterShootingTask")

        from maps.routing.matrix_router.quality_reporter.shooter.lib import shooter

        tvm_config_path = "tvm-config.json"
        with open(tvm_config_path, "w") as tvm_config:
            tvm_config.write(_tvm_config(
                shooter_tvm_id=self.Parameters.shooter_tvm_id,
                shooter_tvm_secret=self._tvm_secret(),
                router_tvm_id=self.Parameters.router_tvm_id))

        os.environ["TVMTOOL_LOCAL_AUTHTOKEN"] = _random_auth_token()

        os.makedirs(_TVMTOOL_CACHE_DIR)

        tvmtool_path = _prepare_tvmtool()
        with sandbox.sdk2.helpers.ProcessLog(self, logger="tvmtool") as tvm_log:
            tvmtool_process = subprocess.Popen(
                [
                    tvmtool_path,
                    "-c", tvm_config_path,
                    "--port", str(_TVMTOOL_PORT),
                    "--cache-dir", _TVMTOOL_CACHE_DIR,
                ],
                stdout=tvm_log.stdout,
                stderr=tvm_log.stderr)
            logging.info("shooting matrix router in testing")
            shooter.shoot(
                router=self.Parameters.router_url,
                tvm_source_alias="shooter",
                tvm_matrix_router_alias="router",
                tvm_mds_alias="mds-stable",
                tvmtool_port=_TVMTOOL_PORT)
            tvmtool_process.terminate()

    def on_break(self, prev_status, status):
        self._send_status_to_juggler("CRIT", status)
        super(MatrixRouterShootingTask, self).on_break(prev_status, status)

    def on_success(self, prev_status):
        self._send_status_to_juggler("OK", "SUCCESS")
        super(MatrixRouterShootingTask, self).on_success(prev_status)

    def on_failure(self, prev_status):
        self._send_status_to_juggler("CRIT", "FAILURE")
        super(MatrixRouterShootingTask, self).on_failure(prev_status)
