import json
import logging
import os
import re
import shutil
import subprocess
import tarfile

from datetime import datetime

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import task
from sandbox.sandboxsdk import environments
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.common.constants import constants as sdk_constants
from sandbox.projects.common.vcs import aapi
from sandbox.projects.maps.common import utils
from sandbox.projects.maps.common.ecstatic_bin import MapsEcstaticToolMixin
from sandbox.projects.maps.common.mapscore_resources import MapsCoreCoverageResource
from sandbox.projects.maps.common.retry import retry

import sandbox.projects.maps.common.jams_common as common

CONFIG_PATH_TEMPLATE = "arcadia:/arc/trunk/arcadia/maps/mobile/server/coverage/layers/{layer}/config.json"

COMMAND_TEMPLATE = "{tool} -o {outPath} -i {geoidPath} " \
    "--tile-size={tileSize} --detailed-zoom-treshold={zoomThreshold} " \
    "--force-min-zoom={forceMinZoom} --eps={epsilon} --layer=geoid " \
    "--region-ids={regionIds} --layer-name={layerName} " \
    "--layer-version={layerVersion} --output-format=mms"

ECSTATIC_TESTING_ENV = 'testing'
ECSTATIC_STABLE_ENV = 'stable'
ECSTATIC_TESTING_BRANCH = 'testing'
ECSTATIC_STABLE_BRANCH = 'stable'
ECSTATIC_ACTIVE_VERSIONS_REGEX = r"[a-zA-Z]*=([^\s]*).*\[(?=.*A)(?=.*R).*\]"

GEOID_PATH = 'geoid/'
GEOID_DATASET_NAME = 'yandex-maps-coverage5-geoid'

DATA_DIR = 'data/'
DESC_DIR = 'desc/'

TOOL_NAME = 'mms2mms'
TOOL_ARCADIA_PATH = 'maps/libs/coverage/tools/mms2mms'
TOOL_ROOT = './'

BUILD_INFO_FILE_NAME = 'build_info.json'
DESCRIPTION_FILE_NAME = 'description.json'

TVM_ID = 2017289
TVM_SECRET_ID = 'sec-01dvdpewvkke97hhz8sg360kya'


class EcstaticClient(MapsEcstaticToolMixin):
    def upload_dataset(self, env, dataset, version, dir, branches):
        logging.info(
            "Uploading {} dataset, version = {} to {}:{}".format(dataset, version, env, branches))
        branches = ['+' + branch for branch in branches]
        self._ecstatic_command(env, ['upload', dataset + '=' + version, dir] + branches)

    def move_dataset(self, env, dataset, version, branches):
        logging.info(
            "Moving {} dataset, version = {} to {}:{}".format(dataset, version, env, branches))
        branches = ['+' + branch for branch in branches]
        self._ecstatic_command(env, ['move', dataset + '=' + version] + branches)

    def download_dataset(self, env, dataset, branch, dir):
        version = self.get_active_version(env, dataset, branch)
        if not version:
            raise errors.TaskFailure(
                "Couldn't find active version of {} dataset in {}".format(dataset, env))
        self._ecstatic_command(env, ['download', dataset + '=' + version, '--out', dir])
        pass

    def get_active_version(self, env, dataset, branch):
        raw_versions = self._ecstatic_command(env, ['versions', '--with-status', dataset, branch])
        versions = re.findall(ECSTATIC_ACTIVE_VERSIONS_REGEX, raw_versions)
        versions.sort()
        return versions[-1]

    def _ecstatic_command(self, env, params):
        try:
            return self._retry_call(lambda *args: self.ecstatic(*args, tvm_id=TVM_ID, tvm_secret_id=TVM_SECRET_ID))(env, params)
        except subprocess.CalledProcessError as ex:
            logging.error('Ecstatic returned {}: {}'.format(
                ex.returncode, ex.output))
            raise errors.TaskFailure(
                "Couldn't execute ecstatic command, code = {}".format(ex.returncode))
        pass

    def _retry_call(self, *args):
        return retry(
            exceptions=subprocess.CalledProcessError,
            tries=5,
            delay=30,
            max_delay=120,
            backoff=2)(*args)


class MapsMobileCoverageBuilder(sdk2.Task, EcstaticClient):
    """
    Task for building coverage data from GeoID with mms2mms utilty
    """

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.SvnEnvironment(),
        )

    class Context(sdk2.Task.Context):
        version = datetime.now().strftime("%Y.%m.%d.%H%M%S")

    class Parameters(sdk2.Task.Parameters):
        layer_name = sdk2.parameters.String('Layer name', required=True)

    def on_prepare(self):
        self.arcadia_revision = aapi.ArcadiaApi.svn_head()
        self.resource = MapsCoreCoverageResource(
            self,
            path="{}.tar.gz".format(self.Parameters.layer_name),
            sandbox_task_id=sdk2.Task.current.id,
            sandbox_task_type=sdk2.Task.current.type.__name__,
            sandbox_task_owner=sdk2.Task.current.owner,
            arcadia_revision=self.arcadia_revision,
            version=self.Context.version,
            description="Maps coverage {} dataset".format(self.Parameters.layer_name)
        )

        config_path = CONFIG_PATH_TEMPLATE.format(layer=self.Parameters.layer_name.lower())
        try:
            self.Context.config = self._config(config_path)
        except:
            raise errors.TaskFailure(
                "Failed loading config for layer {} at {}".format(self.Parameters.layer_name, config_path))

    def on_execute(self):
        self._build_tool()
        self._download_geoid()

        self._make_dir(DATA_DIR)
        self._make_dir(DESC_DIR)

        self._build_datasets()
        self._make_resource(self.resource, [DATA_DIR, DESC_DIR])

        self._upload_datasets(ECSTATIC_TESTING_ENV, [ECSTATIC_TESTING_BRANCH, ECSTATIC_STABLE_BRANCH])
        self._upload_datasets(ECSTATIC_STABLE_ENV, [])

    def on_release(self, params):
        if (params['release_status'] == task.ReleaseStatus.TESTING):
            self._move_datasets(ECSTATIC_STABLE_ENV, [ECSTATIC_TESTING_BRANCH])
        elif (params['release_status'] == task.ReleaseStatus.STABLE):
            self._move_datasets(ECSTATIC_STABLE_ENV, [ECSTATIC_STABLE_BRANCH])
        else:
            raise errors.TaskFailure(
                "Unexpected relese branch: {}".format(params['release_status']))

    def _build_tool(self):
        with arcadia_sdk.mount_arc_path(sdk2.svn.Arcadia.ARCADIA_TRUNK_URL) as revision_root:
            arcadia_sdk.do_build(
                build_system=sdk_constants.DISTBUILD_BUILD_SYSTEM,
                source_root=revision_root,
                targets=[TOOL_ARCADIA_PATH],
                results_dir=TOOL_ROOT,
                clear_build=False,
            )

    def _download_geoid(self):
        self.download_dataset(ECSTATIC_STABLE_ENV, GEOID_DATASET_NAME, ECSTATIC_STABLE_BRANCH, GEOID_PATH)

    def _build_datasets(self):
        logging.info("Building {} coverage, tile size = {}, epsilon = {}".format(
            self.Context.config["layerName"],
            self.Context.config["tileSize"],
            self.Context.config["epsilon"]))

        utils.write_build_info_file(
            DATA_DIR,
            self.arcadia_revision,
            self.Context.dataset_version,
            {"layer_name": self.Context.config["layerName"]})

        command = COMMAND_TEMPLATE.format(
            tool=os.path.join(TOOL_ROOT, TOOL_ARCADIA_PATH, TOOL_NAME),
            outPath=DATA_DIR,
            geoidPath=GEOID_PATH,
            tileSize=self.Context.config["tileSize"],
            zoomThreshold=self.Context.config["detailedZoomThreshold"],
            epsilon=self.Context.config["epsilon"],
            forceMinZoom=self.Context.config["forceMinZoom"],
            regionIds=self._region_ids(self.Context.config),
            layerName=self.Context.config["layerName"],
            layerVersion=self.Context.version)

        res = utils.execute(self, command, "coverage.building")
        if (res != 0):
            raise errors.TaskFailure("mms2mms returned {}".format(res))

        # Copying description and build_info files
        # to the desc/ directory making up a mobile_init dataset.
        desc_file_name = self.Context.config["layerName"] + ".json"
        shutil.copyfile(os.path.join(DATA_DIR, DESCRIPTION_FILE_NAME), os.path.join(DESC_DIR, desc_file_name))
        shutil.copyfile(os.path.join(DATA_DIR, BUILD_INFO_FILE_NAME), os.path.join(DESC_DIR, BUILD_INFO_FILE_NAME))

    def _make_resource(self, resource, data_paths):
        logging.info("Uploading datasets to sandbox...")

        resource_data = sdk2.ResourceData(resource)
        with tarfile.open(str(resource_data.path), 'w:gz') as tar:
            for entry in data_paths:
                tar.add(entry, arcname=entry)
        resource_data.ready()

    def _upload_datasets(self, env, branches):
        datasets = [
            (self.Context.config["dataPackageName"], DATA_DIR),
            (self.Context.config["descPackageName"], DESC_DIR)
        ]
        for package_name, dir_name in datasets:
            self.upload_dataset(
                env,
                package_name,
                self.Context.version,
                dir_name,
                branches)

    def _move_datasets(self, env, branches):
        datasets = [
            self.Context.config["dataPackageName"],
            self.Context.config["descPackageName"]
        ]
        for package_name in datasets:
            self.move_dataset(
                env,
                package_name,
                self.Context.version,
                branches)

    def _source_from_arcadia(self, arcadia_path):
        try:
            call = retry(
                exceptions=subprocess.CalledProcessError,
                tries=5,
                delay=30,
                max_delay=120,
                backoff=2)

            return call(lambda *args: sdk2.svn.Arcadia.cat(*args))(arcadia_path)
        except subprocess.CalledProcessError as ex:
            self.set_info(ex.output)
            logging.error('Arcadia returned {}: {}'.format(
                ex.returncode, ex.output))
            raise errors.TaskFailure(
                "Couldn't get source file from arcadia, code = {}".format(ex.returncode))

    def _config(self, path):
        config_json = self._source_from_arcadia(path)
        return json.loads(config_json)

    def _region_ids(self, config):
        def get_id(region_desc):
            return region_desc["geoid"]
        return ",".join(str(i) for i in map(get_id, config["regions"]))

    def _make_dir(self, dir_name):
        try:
            os.makedirs(dir_name)
        except os.error:
            logging.warn("Directory '{}' already exists".format(dir_name))

    @property
    def release_template(self):
        return sdk2.ReleaseTemplate(
            types=[
                task.ReleaseStatus.STABLE,
                task.ReleaseStatus.TESTING
            ]
        )
