from sandbox import sdk2

from sandbox.projects.maps.MapsNavistopwatchAmmoGen import MapsNavistopwatchAmmoGen
from sandbox.projects.maps.MapsNavistopwatchAmmoGenMt import MapsNavistopwatchAmmoGenMt
from sandbox.projects.masstransit.common.utils import DateParameter
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from sandbox.common.types.task import Status


def get_load_profile(tag):
    load_values = {
        'info_geometry': 'line(1,5000,30m)',
        'default': 'line(1,10000,15m)'
    }
    return load_values.get(
        tag,
        load_values['default']
    )


def get_const_load_profile(tag):
    load_values = {
        'info_descriptor': 'line(1,1200,60s) const(1200,10m)',
        'info_geometry': 'line(1,300,60s) const(300,10m)',
        'mt_info_pb': 'line(1,300,60s) const(300,10m)',
        'default': 'line(1,1500,60s) const(1500,10m)'
    }
    return load_values.get(
        tag,
        load_values['default']
    )


class MapsNavistopwatchShooting(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space =  1024  # 1GB

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        input = sdk2.parameters.Resource('Router requests', required=True)
        navistopwatch_service = sdk2.parameters.String(
            'Navistopwatch service', default='maps_core_navistopwatch_load')
        with sdk2.parameters.Group("/info parameters") as info_group:
            router = sdk2.parameters.String('Router', default='http://core-driving-router.testing.maps.yandex.net')
            formats = sdk2.parameters.List('Ammo formats', default=['geometry', 'descriptor'])
        with sdk2.parameters.Group("/mt_info parameters") as mt_info_group:
            log_date = DateParameter("Date of log to filter", required=False, default=DateParameter.yesterday())
        with sdk2.parameters.Group('Load profile') as load_profile:
            line_load_profile = sdk2.parameters.Bool('line', default=True)
            const_load_profile = sdk2.parameters.Bool('const', default=True)
        with sdk2.parameters.Output:
            lunapark_links = sdk2.parameters.List('Lunapark links', required=True)

    def _notifications_only_on_failures(self):
        notifications = []
        for notification in self.Parameters.notifications:
            notifications.append(sdk2.Notification(
                statuses=[Status.FAILURE, Status.EXCEPTION, Status.TIMEOUT],
                recipients=notification.recipients,
                transport=notification.transport))
        return notifications

    def _create_info_ammo_generator(self, ammo_format):
        description='{} /info ammo generator for task: {}'.format(ammo_format, self.id)
        return MapsNavistopwatchAmmoGen(
            self,
            description=description,
            notifications=self._notifications_only_on_failures(),
            format=ammo_format,
            input=self.Parameters.input,
            router=self.Parameters.router)

    def _create_mt_info_ammo_generator(self):
        return MapsNavistopwatchAmmoGenMt(
            self,
            description='pb /mt_info ammo generator for task: {}'.format(self.id),
            notifications=self._notifications_only_on_failures(),
            log_date=self.Parameters.log_date)

    def _enqueue_shooting(self, ammo, tag, load_profile):
        description = '{} shooting of navistopwatch for task: {}'.format(tag, self.id)
        component='maps-core-navistopwatch_regression_' + tag
        if load_profile == get_const_load_profile(tag):
            component += '_const'
        config_params = [
            ('meta.component', component),
            ('meta.job_name', component),
            ('phantom.rps_schedule', load_profile)
        ]
        if load_profile == get_load_profile(tag):
            config_params.append(('rcassert.pass', '21 22 23 0'))
        shooting_task = ShootViaTankapi(
            self,
            description=description,
            notifications=self._notifications_only_on_failures(),
            nanny_service = self.Parameters.navistopwatch_service,
            config_source='arcadia',
            config_arc_path='maps/routing/router/regression/tank.cfg',
            config_add_parameters=' '.join('-o \'{}={}\''.format(k, v) for (k, v) in config_params),
            use_monitoring=True,
            monitoring_source='arcadia',
            monitoring_arc_path='maps/routing/router/regression/telegraf.xml',
            ammo_source='resource',
            ammo_resource=ammo,
            timeout=5
        ).enqueue()
        self.set_info('Starting subtask: ' + description)
        return shooting_task

    def on_execute(self):
        with self.memoize_stage['create_ammo_tasks']:
            self.Context.tags = {}
            # /info endpoint
            for ammo_format in self.Parameters.formats:
                task = self._create_info_ammo_generator(ammo_format)
                self.Context.tags[unicode(task.id)] = "info_" + ammo_format

            # /mt_info endpoint
            task = self._create_mt_info_ammo_generator()
            self.Context.tags[unicode(task.id)] = "mt_info_pb"
            self.Context.save()

        tasks = []
        tasks += MapsNavistopwatchShooting.find(MapsNavistopwatchAmmoGen, parent=self).limit(0)
        tasks += MapsNavistopwatchShooting.find(MapsNavistopwatchAmmoGenMt, parent=self).limit(0)

        for task in tasks:
            with self.memoize_stage['generate_ammo_' + self.Context.tags[unicode(task.id)]]:
                ammo_generator = task.enqueue()
                self.set_info('Starting subtask: ' + task.Parameters.description)
                self.Context.ammo_generator_id = ammo_generator.id
                self.Context.save()
                raise sdk2.WaitOutput({ammo_generator.id: 'ammo'}, wait_all=True)
            load_profiles = []
            if self.Parameters.line_load_profile:
                load_profiles.append(get_load_profile(self.Context.tags[unicode(task.id)]))
            if self.Parameters.const_load_profile:
                load_profiles.append(get_const_load_profile(self.Context.tags[unicode(task.id)]))
            for load_profile in load_profiles:
                with self.memoize_stage['shoot_' + self.Context.tags[unicode(task.id)] + '_' + load_profile]:
                    ammo_generator = self.find(
                        MapsNavistopwatchAmmoGen, id=self.Context.ammo_generator_id).first()
                    if not ammo_generator:
                        ammo_generator = self.find(
                            MapsNavistopwatchAmmoGenMt, id=self.Context.ammo_generator_id).first()
                    shooting = self._enqueue_shooting(
                        ammo_generator.Parameters.ammo, self.Context.tags[unicode(task.id)], load_profile)
                    raise sdk2.WaitOutput({shooting.id: 'lunapark_link'}, wait_all=True)

        self.Parameters.lunapark_links = [
            shoot.Parameters.lunapark_link for shoot in self.find(ShootViaTankapi)]
