import hashlib
import os
from sandbox import sdk2
import logging
import sandbox.common.types.resource as ctr

from sandbox.projects.maps.MapsBinaryBaseTask import MapsBinaryBaseTask


DEFAULT_AMMO_CONFIG = [
    {'agents': 200, 'counters': {'resources': 50, 'clients': 10}},
    {'agents': 200, 'counters': {'resources': 10, 'clients': 50}},
    {'agents': 100, 'counters': {'resources': 5, 'clients': 500}}
]
INTERLEAVE_CHUNKS_COUNT = 10


class MapsRatelimiter2Ammo(sdk2.Resource):
    name = 'MAPS_RATELIMITER2_AMMO'
    pass


class MapsRatelimiter2AmmoGen(MapsBinaryBaseTask):

    name = 'MAPS_RATELIMITER2_AMMO_GENERATOR'

    class Requirements(MapsBinaryBaseTask.Requirements):
        pass

    class Parameters(MapsBinaryBaseTask.Parameters):
        binary_name = MapsBinaryBaseTask.Parameters.binary_name(default='gen-server-ammo')
        installation = MapsBinaryBaseTask.Parameters.installation(default='stable')
        version_file_path = MapsBinaryBaseTask.Parameters.version_file_path(
            default='maps/infra/ratelimiter2/regression/gen-server-ammo/binary_version.json')

        ammo_profile = sdk2.parameters.JSON('Ammo generation profile', default=DEFAULT_AMMO_CONFIG)

        protocol_v2 = sdk2.parameters.Bool('V2 protocol', default=False)
        with protocol_v2.value[False]:
            server_hostname = sdk2.parameters.String(
                'Ratelimiter server',
                default='core-ratelimiter-load.common.testing.maps.yandex.net',
                description='Used to query limits version'
            )

        with sdk2.parameters.Output:
            ammo = sdk2.parameters.Resource('Ammo', required=True)

    def generate_ammo_data(self, outfile):
        profile = self.Parameters.ammo_profile
        seeds = [chr(ord('A') +i ) for i in range(len(profile))]

        for i, seed in enumerate(seeds):
            filename = '{}.ammo'.format(seed)
            self.set_info('Generating {}'.format(profile[i]))

            options = dict(profile[i]['counters'].items() + {
                'agents': profile[i]['agents'],
                'seed': seed,
                'output': filename}.items())
            if self.Parameters.protocol_v2:
                options['v2-split'] = 0  # TODO: >0 to emulate cluster limits split
            else:
                options['server'] = self.Parameters.server_hostname

            self.run_binary(options=options)

            logging.info('{} splitting into {} chunks'.format(filename, INTERLEAVE_CHUNKS_COUNT))
            os.system('split -d -n {count} {file} {prefix}'.format(
                count=INTERLEAVE_CHUNKS_COUNT, file=filename, prefix=filename+'.chunk'))

        interleave = ' '.join([
            '{}.ammo.chunk{:02}'.format(seed, chunk)
            for chunk in range(INTERLEAVE_CHUNKS_COUNT) for seed in seeds
        ])
        logging.info('Interleaving files: {} '.format(interleave))
        os.system('cat {} > {}'.format(interleave, outfile))

    def on_execute(self):
        # generate ammo
        self.generate_ammo_data('output.ammo')

        # calc md5 of new ammo
        with open('output.ammo', 'rb') as ammo_file:
            new_ammo_md5 = hashlib.md5(ammo_file.read()).hexdigest()

        self.set_info('New ammo md5:{}'.format(new_ammo_md5))

        # 10 last previously generated ammo resources
        ammo_history = MapsRatelimiter2Ammo.find(state=ctr.State.READY).order(-sdk2.Resource.id).limit(10)
        logging.info('Ammo history lookup: {}'.format([(task, task.md5) for task in ammo_history]))

        # lookup same ammo in history
        ammo_history = [a for a in ammo_history if a.md5 == new_ammo_md5]
        ammo = ammo_history[0] if ammo_history else None

        # return ammo from history if found same
        if ammo:
            self.set_info('Reuse ammo resource <a href="{}">{}</a> md5:{}'.format(ammo.url, ammo.id, ammo.md5), do_escape=False)
        else:
            ammo = MapsRatelimiter2Ammo(self, 'MapsRatelimiter2Ammo', 'output.ammo')  # new resource
            ammo_data = sdk2.ResourceData(ammo)
            ammo_data.ready()
            self.set_info('New ammo resource <a href="{}">{}</a> md5:{}'.format(ammo.url, ammo.id, new_ammo_md5), do_escape=False)

        self.Parameters.ammo = ammo
