import jinja2
import random
import os
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.maps.MapsRatelimiter2AmmoGen import MapsRatelimiter2AmmoGen, DEFAULT_AMMO_CONFIG
from sandbox.projects.tank.Firestarter import Firestarter
from sandbox.projects.tank.ShootViaTankapi import detect_tank


class MapsRatelimiter2Shooter(sdk2.Task):

    name = 'MAPS_RATELIMITER2_SHOOTER'

    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 1024  # 1GB guarantee

    class Parameters(sdk2.Task.Parameters):
        target_nanny_service = sdk2.parameters.String('Target nanny service',
                                                      default='nanny:maps_core_ratelimiter_load')

        protocol_v1 = sdk2.parameters.Bool('Shoot protocol V1', default=True, description='/counters/sync')
        with protocol_v1.value[True]:
            shooting_profile_imbalance_v1 = sdk2.parameters.String(
                'Imbalance RPS profile',
                default='[{duration: 15m, from: 1, to: 3500, type: line}]')
            shooting_profile_constant_v1 = sdk2.parameters.String(
                'Constant RPS profile',
                default='[{duration: 60s, from: 1, to: 2000, type: line}, {duration: 10m, ops: 2000, type: const}]'
            )
            ammo_profile_v1 = sdk2.parameters.JSON('Ammo generation profile', default=DEFAULT_AMMO_CONFIG)

        protocol_v2 = sdk2.parameters.Bool('Shoot protocol V2', default=True, description='/v2/counters/sync')
        with protocol_v2.value[True]:
            shooting_profile_imbalance_v2 = sdk2.parameters.String(
                'Imbalance RPS profile',
                default='[{duration: 15m, from: 1, to: 3500, type: line}]')
            shooting_profile_constant_v2 = sdk2.parameters.String(
                'Constant RPS profile',
                default='[{duration: 60s, from: 1, to: 2000, type: line}, {duration: 10m, ops: 2000, type: const}]'
            )
            ammo_profile_v2 = sdk2.parameters.JSON('Ammo generation profile', default=DEFAULT_AMMO_CONFIG)

        shooting_config_template = sdk2.parameters.String(
            'Shooting config template',
            default='maps/infra/ratelimiter2/regression/shoot-ratelimiter-server-template.yaml',
            description='Arcadia path')

        st_ticket = sdk2.parameters.StrictString(
            'Startrek ticket', default='GEOINFRA-410', regexp='[A-Z]{2,}-[0-9]+')

    def generate_config(self, config_template, **kwargs):
        return jinja2.Template(config_template).render(
            st_ticket=self.Parameters.st_ticket,
            **kwargs
        )

    def _notifications_only_on_failures(self):
        notifications = []
        for notification in self.Parameters.notifications:
            notifications.append(sdk2.Notification(
                statuses=[ctt.Status.FAILURE, ctt.Status.EXCEPTION, ctt.Status.TIMEOUT],
                recipients=notification.recipients,
                transport=notification.transport))
        return notifications

    def _run_ammo_gen_task(self, title, **kwargs):
        ammo_gen_task = MapsRatelimiter2AmmoGen(
            self,
            description='Ratelimiter {} ammo generation'.format(title),
            notifications=self._notifications_only_on_failures(),
            **kwargs
        ).enqueue()

        self.set_info(
            'Running {title} ammo generator subtask: <a href="https://sandbox.yandex-team.ru/task/{id}">{id}</a>'.format(
                title=title, id=ammo_gen_task.id),
            do_escape=False)
        return ammo_gen_task

    def _run_shooting_task(self, title, config):
        shooting_task = Firestarter(
            self,
            description='Ratelimiter {} shooting'.format(title),
            notifications=self._notifications_only_on_failures(),
            tank_config=config,
        )

        # Shooting subtasks inherit parent semaphores
        if self.Requirements.semaphores:
            shooting_task.Requirements.semaphores = self.Requirements.semaphores

        shooting_task.save().enqueue()

        self.set_info(
            'Running {title} shooting subtask: <a href="https://sandbox.yandex-team.ru/task/{id}">{id}</a>'.format(
                title=title, id=shooting_task.id),
            do_escape=False)
        return shooting_task

    class Context(sdk2.Task.Context):
        shooters = []
    
    def generate_lunapark_link(self):
        return 'https://lunapark.yandex-team.ru/{}'.format(sdk2.Task[self.Context.shooters[-1]].Parameters.lunapark_id)
    
    def on_execute(self):
        with self.memoize_stage.prepare:
            # Select target and tanks
            target_nanny = self.Parameters.target_nanny_service
            target_nanny = target_nanny[target_nanny.startswith('nanny:') and len('nanny:') or 0:]
            tank_finder = detect_tank.TankFinder(target_nanny_service=target_nanny,
                                                 tanks_nanny_services=['maps_core_tanks_load'])
            tank_finder.collect_info()
            target = random.choice(tank_finder.targets)
            self.Context.target = target.host

            self.set_info('Selected target: {}'.format(self.Context.target))

            # Load shooting config template
            with arcadia_sdk.mount_arc_path('arcadia-arc:/#trunk') as arcadia:
                with open(os.path.join(arcadia, self.Parameters.shooting_config_template)) as config_file:
                    self.Context.config_template = config_file.read()

            # Run ammo generation subtasks
            ammogen_tasks = []
            if self.Parameters.protocol_v1:
                # NB: V1 queries limits through 'core-ratelimiter-load.common.testing.maps.yandex.net' (server_hostname parameter default)
                # We can't use target hostname (no direct network access from sandbox to instances)
                task = self._run_ammo_gen_task(title='V1', ammo_profile=self.Parameters.ammo_profile_v1)
                self.Context.ammogen_v1_task_id = task.id
                ammogen_tasks.append(task)
            if self.Parameters.protocol_v2:
                task = self._run_ammo_gen_task(title='V2', protocol_v2=True,
                                               ammo_profile=self.Parameters.ammo_profile_v2)
                self.Context.ammogen_v2_task_id = task.id
                ammogen_tasks.append(task)

            raise sdk2.WaitTask(ammogen_tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

        ammo_v1 = sdk2.Task[self.Context.ammogen_v1_task_id].Parameters.ammo \
            if self.Context.ammogen_v1_task_id else None
        ammo_v2 = sdk2.Task[self.Context.ammogen_v2_task_id].Parameters.ammo \
            if self.Context.ammogen_v2_task_id else None

        if self.Parameters.protocol_v1:
            with self.memoize_stage.shooting_imbalance_v1:
                config = self.generate_config(
                    self.Context.config_template,
                    target_hostname=self.Context.target,
                    shooting_profile=self.Parameters.shooting_profile_imbalance_v1,
                    ammo_url=ammo_v1.http_proxy,
                    shooting_component='maps-core-ratelimiter server imbalance'
                )
                shooter_imbalance = self._run_shooting_task(
                    title='V1 imbalance', config=config)
                self.Context.shooters.append(shooter_imbalance.id)
                raise sdk2.WaitTask(shooter_imbalance.id, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

            if len(self.Context.shooters) == 1:
                self.set_info(
                    'V1 shooting finished, Lunapark link: <a href="{link}">{link}</a>'.format(
                        link=self.generate_lunapark_link()),
                    do_escape=False)

            with self.memoize_stage.shooting_constant_v1:
                config = self.generate_config(
                    self.Context.config_template,
                    target_hostname=self.Context.target,
                    shooting_profile=self.Parameters.shooting_profile_constant_v1,
                    ammo_url=ammo_v1.http_proxy,
                    shooting_component='maps-core-ratelimiter server rps constant'
                )
                shooter_constant = self._run_shooting_task(
                    title='V1 constant', config=config)
                self.Context.shooters.append(shooter_constant.id)
                raise sdk2.WaitTask(shooter_constant.id, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

            if len(self.Context.shooters) == 2:
                self.set_info(
                    'V1 shooting finished, Lunapark link: <a href="{link}">{link}</a>'.format(
                        link=self.generate_lunapark_link()),
                    do_escape=False)

        if self.Parameters.protocol_v2:
            with self.memoize_stage.shooting_imbalance_v2:
                config = self.generate_config(
                    self.Context.config_template,
                    target_hostname=self.Context.target,
                    shooting_profile=self.Parameters.shooting_profile_imbalance_v2,
                    ammo_url=ammo_v2.http_proxy,
                    shooting_component='maps-core-ratelimiter server V2 imbalance'
                )
                shooter_imbalance = self._run_shooting_task(
                    title='V2 imbalance', config=config)
                self.Context.shooters.append(shooter_imbalance.id)
                raise sdk2.WaitTask(shooter_imbalance.id, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

            if len(self.Context.shooters) in [1,3]:
                self.set_info(
                    'V2 shooting finished, Lunapark link: <a href="{link}">{link}</a>'.format(
                        link=self.generate_lunapark_link()),
                    do_escape=False)

            with self.memoize_stage.shooting_constant_v2:
                config = self.generate_config(
                    self.Context.config_template,
                    target_hostname=self.Context.target,
                    shooting_profile=self.Parameters.shooting_profile_constant_v2,
                    ammo_url=ammo_v2.http_proxy,
                    shooting_component='maps-core-ratelimiter server V2 rps constant'
                )
                shooter_constant = self._run_shooting_task(
                    title='V2 constant', config=config)
                self.Context.shooters.append(shooter_constant.id)
                raise sdk2.WaitTask(shooter_constant.id, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

            if len(self.Context.shooters) in [2,4]:
                self.set_info(
                    'V2 shooting finished, Lunapark link: <a href="{link}">{link}</a>'.format(
                        link=self.generate_lunapark_link()),
                    do_escape=False)

        shooters = [sdk2.Task[task_id] for task_id in self.Context.shooters]

        # Failure if any of shootings failed
        failed_shooters = [task for task in shooters if task.status not in ctt.Status.Group.SUCCEED]
        if failed_shooters:
            raise TaskFailure('Shooting tasks reported: {}'.format(
                ', '.join([task.status for task in failed_shooters])))
