import os
import datetime
import subprocess

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
import sandbox.projects.maps.carparks.common.utils as utils

from sandbox.projects.maps.common.ecstatic_bin import MapsEcstaticToolMixin


DATASET_NAME = 'yandex-maps-realty-jams'

TESTING_TVM_ID = 2018446
TESTING_SECRET_TVM_ID = 'sec-01e0skwxn681sdwjqst558p8jp'
PRODUCTION_TVM_ID = 2018444
PRODUCTION_SECRET_TVM_ID = 'sec-01e0sg6ma6nk4z1fp09qk0bnyx'


class MapsRealtyJams(MapsEcstaticToolMixin, sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1

    class Parameters(sdk2.Parameters):

        vault_owner = sdk2.parameters.String(
            'Sandbox Vault owner',
            required=True,
            default='robot-maps-carparks')

        yql_token_vault_name = sdk2.parameters.String(
            'YQL token Sandbox Vault name',
            required=True,
            default='YQL_TOKEN')

        yt_token_vault_name = sdk2.parameters.String(
            'YT token Sandbox Vault name',
            required=True,
            default='YT_TOKEN')

        realty_jams_generator_binary = sdk2.parameters.Resource(
            'Realty jams generator binary',
            required=True)

        environment = sdk2.parameters.String(
            'Ecstatic environment',
            default_value='stable')

        config_arc_path = sdk2.parameters.String(
            "Path to realty jams config",
            required=True,
            description='for example: maps/realty/jams_generator/'
                        'config/config.testing.json'
        )

    def _local_config_path(self):
        return os.getcwd() + '/config.json'

    def _download_config(self):
        arcadia_url = sdk2.svn.Arcadia.trunk_url(
            self.Parameters.config_arc_path
        )

        sdk2.svn.Arcadia.export(
            arcadia_url,
            self._local_config_path()
        )

    def _prepare_tokens(self):
        os.environ['YQL_TOKEN'] = sdk2.Vault.data(
            self.Parameters.vault_owner,
            self.Parameters.yql_token_vault_name)

        os.environ['YT_TOKEN'] = sdk2.Vault.data(
            self.Parameters.vault_owner,
            self.Parameters.yt_token_vault_name)

    def _generate_realty_jams(self):
        os.makedirs('data')
        output_file = './data/realty_jams.fb'
        self._download_config()
        utils.run(self.Parameters.realty_jams_generator_binary,
                  '--output', output_file,
                  '--config', self._local_config_path())

    def _ecstatic_upload(self):
        version = datetime.datetime.utcnow().strftime('%Y-%m-%d-%H-%M')

        if self.Parameters.environment == 'stable':
            tvm_id = PRODUCTION_TVM_ID
            tvm_secret_id = PRODUCTION_SECRET_TVM_ID
        else:
            tvm_id = TESTING_TVM_ID
            tvm_secret_id = TESTING_SECRET_TVM_ID
        try:
            self.ecstatic(
                self.Parameters.environment,
                ['upload',
                 DATASET_NAME + '=' + version,
                 'data',
                 '+stable'],
                tvm_id=tvm_id,
                tvm_secret_id=tvm_secret_id)
        except subprocess.CalledProcessError as e:
            self.set_info(e.output)
            raise TaskFailure('Ecstatic returned ' + str(e.returncode))

    def on_execute(self):
        self._prepare_tokens()
        self._generate_realty_jams()
        self._ecstatic_upload()
