import os

import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common.errors import TaskFailure, TaskStop


class MapsStatisticsCubeMaker(sdk2.Task, object):
    """
    Executes StatisticsCubeMaker binary with given params
    """

    fail_on_any_error = True

    class Requirements(sdk2.Requirements):
        cores = 1  # 1 cores or less
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches


    class Parameters(sdk2.Parameters):
        with sdk2.parameters.String("Environment", multiline=True, default='testing') as env_parameter:
            env_parameter.values['testing'] = 'testing'
            env_parameter.values['production'] = 'production'


    def on_enqueue(self):
        running_tasks = sdk2.Task.find(
            MapsStatisticsCubeMaker,
            status=ctt.Status.Group.QUEUE + ctt.Status.Group.EXECUTE + ctt.Status.Group.WAIT,
            input_parameters={
                'env_parameter': self.Parameters.env_parameter
            }
        ).limit(2)

        ids = set([task.id for task in running_tasks])
        ids.discard(self.id)

        if len(ids) > 0:
            raise TaskStop('There are other tasks enqueued/executing with same parmeters. Stopping.')


    def on_execute(self):
        attrs = {}
        if self.Parameters.env_parameter == 'production':
            attrs['released'] = 'stable'

        resource = sdk2.Resource["MAPS_STATISTICS_CUBE_MAKER_EXECUTABLE"].find(
            state=ctr.State.READY,
            attrs=attrs
        ).order(-sdk2.Resource.id).first()

        resource_data = sdk2.ResourceData(resource)
        resource_path = resource_data.path
        binary = str(resource_path / 'cube_maker' / 'bin' / 'cube_maker')

        command = [binary]
        command.append('--environment')
        command.append(self.Parameters.env_parameter)

        with sdk2.helpers.ProcessLog(self, logger="statistics_cube_maker") as pl:
            pl.logger.propagate = 1
            my_env = os.environ.copy()
            my_env['YT_TOKEN'] = sdk2.Vault.data("ROBOT_STATS_NAVI_YT_TOKEN")
            if sp.Popen(command, shell=False, env=my_env, stdout=pl.stdout, stderr=sp.STDOUT).wait() != 0:
                raise TaskFailure('Statistics_cube_maker exited with error. See logs for details.')
