import os

import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common.errors import TaskFailure, TaskStop

YT_POOL = 'maps-analytics-privileged'
YT_POOL_TAXIMETER = 'maps-analytics'
YT_TOKEN_SECRET_ID = "sec-01dh94knca8ad1gha9cypwq42s"


class MapsStatisticsLogPreprocessorV2(sdk2.Task, object):
    """
    Executes StatisticsLogPreprocessor binary with given params (for use with Reactor)
    """

    fail_on_any_error = True

    class Requirements(sdk2.Requirements):
        cores = 1  # 1 cores or less
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        date_parameter = sdk2.parameters.String("Date", required=True)
        with sdk2.parameters.String("Component", multiline=True, default='navigator') as component_parameter:
            component_parameter.values['navigator'] = 'navigator'
            component_parameter.values['maps'] = 'maps'
            component_parameter.values['auto'] = 'auto'
            component_parameter.values['taximeter'] = 'taximeter'
        with sdk2.parameters.String("Environment", multiline=True, default='testing') as env_parameter:
            env_parameter.values['testing'] = 'testing'
            env_parameter.values['production'] = 'production'
        force_parameter = sdk2.parameters.Bool("Force recalculate", default=False)

    def on_enqueue(self):
        running_tasks = sdk2.Task.find(
            MapsStatisticsLogPreprocessorV2,
            status=ctt.Status.Group.QUEUE + ctt.Status.Group.EXECUTE + ctt.Status.Group.WAIT,
            input_parameters={
                'component_parameter': self.Parameters.component_parameter,
                'date_parameter': self.Parameters.date_parameter,
                'env_parameter': self.Parameters.env_parameter,
                'force_parameter': self.Parameters.force_parameter
            }
        ).limit(10)

        ids = set([task.id for task in running_tasks])
        ids.discard(self.id)

        if len(ids) > 0:
            raise Exception('There are other tasks enqueued/executing with same parmeters. Stopping.')

    def on_execute(self):
        attrs = {}
        if self.Parameters.env_parameter == 'production':
            attrs['released'] = 'stable'

        resource = sdk2.Resource["MAPS_STATISTICS_LOG_PREPROCESSOR_EXECUTABLE_V_2"].find(
            state=ctr.State.READY,
            attrs=attrs
        ).order(-sdk2.Resource.id).first()

        resource_data = sdk2.ResourceData(resource)
        resource_path = resource_data.path
        binary = str(resource_path / 'log_preprocessor' / 'bin' / 'log_preprocessor')

        command = [binary]
        command.append('--date')
        command.append(self.Parameters.date_parameter)
        command.append('--env')
        command.append(self.Parameters.env_parameter)
        command.append('--component')
        command.append(self.Parameters.component_parameter)
        if self.Parameters.force_parameter is True:
            command.append('--force')

        with sdk2.helpers.ProcessLog(self, logger="log_preprocessor") as pl:
            pl.logger.propagate = 1
            my_env = os.environ.copy()
            my_env['YT_TOKEN'] = sdk2.yav.Secret(YT_TOKEN_SECRET_ID).data()["YT_TOKEN"]
            my_env['YT_POOL'] = YT_POOL if self.Parameters.component_parameter != 'taximeter' else YT_POOL_TAXIMETER
            if sp.Popen(command, shell=False, env=my_env, stdout=pl.stdout, stderr=sp.STDOUT).wait() != 0:
                raise TaskFailure('log_preprocessor exited with error. See logs for details.')
