import os

import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common.errors import TaskFailure, TaskStop

YT_POOL = 'maps-core-navi-statistics'


class MapsStatisticsMetrics(sdk2.Task, object):
    """
    Executes StatisticsMetrics binary with given params
    """

    fail_on_any_error = True

    class Requirements(sdk2.Requirements):
        cores = 1  # 1 cores or less
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches


    class Parameters(sdk2.Parameters):
        actor_parameter = sdk2.parameters.String("Actor", required=True)
        with sdk2.parameters.String("Component", multiline=True, default='navigator') as component_parameter:
            component_parameter.values['navigator'] = 'navigator'
            component_parameter.values['maps'] = 'maps'
            component_parameter.values['auto'] = 'auto'

        with sdk2.parameters.String("Period", multiline=True, default='Day') as period_parameter:
            period_parameter.values['Day'] = 'Day'
            period_parameter.values['Week'] = 'Week'
            period_parameter.values['Month'] = 'Month'

        with sdk2.parameters.String("Environment", multiline=True, default='testing') as env_parameter:
            env_parameter.values['testing'] = 'testing'
            env_parameter.values['production'] = 'production'


    def on_enqueue(self):
        running_tasks = sdk2.Task.find(
            MapsStatisticsMetrics,
            status=ctt.Status.Group.QUEUE + ctt.Status.Group.EXECUTE + ctt.Status.Group.WAIT,
            input_parameters={
                'actor_parameter': self.Parameters.actor_parameter,
                'period_parameter': self.Parameters.period_parameter,
                'component_parameter': self.Parameters.component_parameter,
                'env_parameter': self.Parameters.env_parameter
            }
        ).limit(10)

        ids = set([task.id for task in running_tasks])
        ids.discard(self.id)

        if len(ids) > 0:
            raise TaskStop('There are other tasks enqueued/executing with same parmeters. Stopping.')


    def on_execute(self):
        attrs = {}
        if self.Parameters.env_parameter == 'production':
            attrs['released'] = 'stable'

        resource  = sdk2.Resource["MAPS_STATISTICS_METRICS_EXECUTABLE"].find(
            state=ctr.State.READY,
            attrs=attrs
        ).order(-sdk2.Resource.id).first()

        resource_data = sdk2.ResourceData(resource)
        resource_path = resource_data.path
        binary = str(resource_path / 'metrics' / 'bin' / 'metrics')

        command = [binary]
        command.append('--actor')
        command.append(self.Parameters.actor_parameter)
        command.append('--component')
        command.append(self.Parameters.component_parameter)
        command.append('--period')
        command.append(self.Parameters.period_parameter)
        command.append('--check-freshness')
        command.append('True')

        with sdk2.helpers.ProcessLog(self, logger="statistics_metrics_daily") as pl:
            pl.logger.propagate = 1
            my_env = os.environ.copy()
            my_env['ENV_TYPE'] = self.Parameters.env_parameter
            my_env['YT_TOKEN'] = sdk2.Vault.data("ROBOT_STATS_NAVI_YT_TOKEN")
            my_env['YT_POOL'] = YT_POOL
            if sp.Popen(command, shell=False, env=my_env, stdout=pl.stdout, stderr=sp.STDOUT).wait() != 0:
                raise TaskFailure('Statistics_metrics_daily exited with error. See logs for details.')
