import logging
import requests
import time

from os import environ

logger = logging.getLogger()

TOLOKA_API_URL = 'https://{}toloka.yandex.ru/api'
TOLOKA_COMISSION_RATE = 0.2
MIN_TOLOKA_COMISSION = 0.005


def retry(request, parse_body=True, n_retries=5):
    for i in range(n_retries):
        try:
            response = request()
            logger.debug(
                'retry %d finished with status %d. response body: "%s"',
                i,
                response.status_code,
                response.text
            )
            response.raise_for_status()
            return response.json() if parse_body else response
        except Exception as e:
            logger.debug('retry %d encountered error: %s', i, str(e))
            if i + 1 == n_retries:
                raise
            continue


def toloka_headers():
    return {
        'Authorization': 'OAuth {}'.format(environ['TOLOKA_TOKEN']),
        'Content-Type': 'application/JSON',
    }


def open_toloka_pool(pool_id, sandbox_toloka=False):
    api_url = TOLOKA_API_URL.format('sandbox.' if sandbox_toloka else '')
    response = retry(
        lambda: requests.post(
            '{}/v1/pools/{}/open'.format(api_url, pool_id),
            headers=toloka_headers()
        ),
        parse_body=False,
    )


def unfinished_task_count(pool_id, sandbox_toloka=False):
    api_url = TOLOKA_API_URL.format('sandbox.' if sandbox_toloka else '')

    def get_pool_statistics(stat_names):
        op_ids = []
        for name in stat_names:
            logger.debug('requesting %s info for pool %s', name, pool_id)
            response = retry(lambda: requests.post(
                '{}/staging/analytics-2'.format(api_url),
                headers=toloka_headers(),
                json=[{'subject': 'POOL', 'subject_id': pool_id, 'name': name}],
            ))
            op_ids.append(response['id'])
        while True:
            time.sleep(10)  # 10s
            results = []
            for id in op_ids:
                response = retry(lambda: requests.get(
                    '{}/v1/operations/{}'.format(api_url, id),
                    headers=toloka_headers()
                ))
                if response['status'] != 'PENDING' and response['status'] != 'RUNNING':
                    results.append(response['details']['value'][0]['result'])
            if len(results) == len(stat_names):
                break
        return results

    pool_stats = get_pool_statistics([
        'estimated_assignments_count',
        'approved_assignments_count',
        'rejected_assignments_count',
    ])
    total_task_pages = pool_stats[0]['value']
    accepted_task_pages = pool_stats[1]
    rejected_task_pages = pool_stats[2]
    return total_task_pages - accepted_task_pages - rejected_task_pages


def check_toloka_pool_finished(pool_id, sandbox_toloka=False):
    return unfinished_task_count(pool_id, sandbox_toloka) == 0


def check_have_money_for_task(pool_id, sandbox_toloka=False, max_spendings=float('inf')):
    api_url = TOLOKA_API_URL.format('sandbox.' if sandbox_toloka else '')
    response = retry(lambda: requests.get(
        '{}/v1/requester'.format(api_url),
        headers=toloka_headers()
    ))
    balance = float(response['balance'])

    response = retry(lambda: requests.get(
        '{}/v1/pools/{}'.format(api_url, pool_id),
        headers=toloka_headers()
    ))
    page_price = float(response['reward_per_assignment'])
    page_price += max(round(page_price * TOLOKA_COMISSION_RATE, 3), MIN_TOLOKA_COMISSION)

    total_price = unfinished_task_count(pool_id, sandbox_toloka) * page_price
    if total_price > max_spendings:
        raise ValueError('Spendings are too big! Total price is {}, but maximum allowed is {}'.format(total_price, max_spendings))

    return total_price < balance
