import logging
import os
import shutil
import subprocess

from sandbox import sdk2
from sandbox.projects.maps.common.juggler_alerts import (
    TaskJugglerReportWithParameters
)
from sandbox.projects.maps.common.latest_resources import find_latest_resource


class MapsUpdateDynamicPoiDummySourceExecutable(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class MapsUpdateDynamicPoiDummySource(TaskJugglerReportWithParameters):
    # requrements for MULTISLOT tag
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 8 * 1024  # 8 GB

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(TaskJugglerReportWithParameters.Parameters):
        kill_timeout = 30 * 60  # s

        with sdk2.parameters.RadioGroup('Environment') as environment:
            environment.values["testing"] = environment.Value("testing", default=True)
            environment.values["stable"] = environment.Value("stable")

        yt_vault = sdk2.parameters.Vault('YT vault', required=True)
        orgs_per_user = sdk2.parameters.Integer(
            'How many orgs per user to generate',
            default=100,
            required=True,
        )
        exp_prefix = sdk2.parameters.String(
            'Prefix for all experiments tags (even the main one)',
            default='',
        )
        exps_size = sdk2.parameters.Integer(
            'How many experiments should be in data',
            default=10,
            required=True,
        )
        test_users_table = sdk2.parameters.String(
            'Path to YT table with test users',
            default='//home/maps/poi/personalized_poi/test-users/test_users_3_10',
            required=True,
        )
        orgs_table = sdk2.parameters.String(
            'Path to YT table with valid orgs',
            default='//home/maps/poi/personalized_poi/test-users/top_khamovniki',
            required=True,
        )
        output_table = sdk2.parameters.String(
            'Output table with dummy source',
            default='//home/maps/poi/personalized_poi/dynamic_poi/source/dummy/latest',
            required=True,
        )
        dummy_source_executable = sdk2.parameters.Resource(
            'Sandbox resource ID of the executable to update dummy source',
            resource_type=MapsUpdateDynamicPoiDummySourceExecutable,
            required=False,
        )

    def _ensure_latest_resources_used(self):
        self.dummy_source_executable = self.Parameters.dummy_source_executable
        if not self.dummy_source_executable:
            self.dummy_source_executable = find_latest_resource(
                MapsUpdateDynamicPoiDummySourceExecutable,
                self.Parameters.environment)

        logging.info(
            'Working in %s environment', self.Parameters.environment)
        logging.info(
            'Using MapsUpdateDynamicPoiDummySourceExecutable %s',
            self.dummy_source_executable.id)

    def _load_executable(self, resource_id, name):
        resource = sdk2.Resource[resource_id]
        resource_data = sdk2.ResourceData(resource)
        executable_path = './{}'.format(name)
        shutil.copyfile(str(resource_data.path), executable_path)
        subprocess.check_call('chmod +x {}'.format(executable_path), shell=True)

    def on_execute(self):
        self._ensure_latest_resources_used()

        os.environ['YT_TOKEN'] = self.Parameters.yt_vault.data()
        os.environ['YT_PROXY'] = 'hahn'

        self._load_executable(self.dummy_source_executable.id, 'update')
        with self.memoize_stage.update_dummy_source:
            cmd = [
                './update',
                '--orgs-per-user',
                str(self.Parameters.orgs_per_user),
                '--exps-size',
                str(self.Parameters.exps_size),
                '--test-users',
                str(self.Parameters.test_users_table),
                '--orgs-table',
                str(self.Parameters.orgs_table),
                '--output-table',
                str(self.Parameters.output_table),
            ]
            if self.Parameters.exp_prefix:
                cmd.extend([
                    '--exp-prefix',
                    str(self.Parameters.exp_prefix),
                ])
            logging.info('Running command %s', ' '.join(cmd))
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('update dummy source')) as pl:
                subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)
