from sandbox.projects.maps.common.juggler_alerts import (
    TaskJugglerReportWithParameters
)

import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.maps.common.ecstatic_bin import MapsEcstaticToolMixin
from sandbox.projects.maps.common.latest_resources import find_latest_resource

import logging
import os
import subprocess
import shutil

from datetime import datetime


DATASET_NAME = 'yandex-maps-perspoi-dynamic'
BRANCH = 'stable'
TVM_ID_DATATESTING = 2019479
TVM_SECRET_ID_DATATESTING = 'sec-01e4qzwj5j9ybygg02zc5f41d5'
TVM_ID_STABLE = 2019477
TVM_SECRET_ID_STABLE = 'sec-01e4qzw08mx9y9jf19ndy4n93k'

DATETIME_FORMAT = '%y-%m-%d-%H-%M-%S'


class DynamicPoiCalculatorExecutable(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class UploadDynamicPoiToEcstatic(MapsEcstaticToolMixin, TaskJugglerReportWithParameters):
    """Packs dynamic POI tables into fbs and uploads to ecstatic"""
    class Requirements(sdk2.Task.Requirements):
        ram = 4 << 10  # 10 GB
        disk_space = 4 << 10
        client_tags = ctc.Tag.GENERIC & (ctc.Tag.SAS | ctc.Tag.VLA)
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(TaskJugglerReportWithParameters.Parameters):
        kill_timeout = 30 * 60
        owner = 'MAPS'

        with sdk2.parameters.RadioGroup('Environment') as environment:
            environment.values["testing"] = environment.Value("testing", default=True)
            environment.values["stable"] = environment.Value("stable")

        with sdk2.parameters.Group('Dataset version') as dataset_version_parameters:
            forced_datetime = sdk2.parameters.String(
                'Some forced datetime if really needed, {}'.format(DATETIME_FORMAT),
                default='',
                required=False
            )
            version = sdk2.parameters.Integer('version', default=0)

        with sdk2.parameters.Group('Packer parameters') as packer_parameters:
            yt_token = sdk2.parameters.Vault('yt token vault', required=True)
            juggler_token = sdk2.parameters.Vault('juggler token vault')
            dynamic_poi_sources = sdk2.parameters.Dict(
                'Sources names with paths',
                default={'dummy': '//home/maps/poi/personalized_poi/dynamic_poi/source/dummy/latest'},
            )
            output_yt_table_prefix = sdk2.parameters.String(
                'Yt folder where debug YT tables will be placed',
                default='//home/maps/poi/personalized_poi/dynamic_poi/ecstatic_history',
                required=True
            )
            resources_history_yt_file = sdk2.parameters.String(
                'Resources history YT file',
                default='//home/maps/poi/personalized_poi/dynamic_poi/resources_history',
                required=True
            )
            calculator_binary = sdk2.parameters.Resource(
                'Sandbox resource ID for executor binary',
                resource_type=DynamicPoiCalculatorExecutable,
                required=False
            )

    def _load_executable(self, resource_id, name):
        resource = sdk2.Resource[resource_id]
        resource_data = sdk2.ResourceData(resource)
        executable_path = './{}'.format(name)
        shutil.copyfile(str(resource_data.path), executable_path)
        subprocess.check_call('chmod +x {}'.format(executable_path), shell=True)

    def _ensure_latest_resources_used(self):
        self.calculator_binary = self.Parameters.calculator_binary
        if not self.calculator_binary:
            self.calculator_binary = find_latest_resource(
                DynamicPoiCalculatorExecutable, self.Parameters.environment)

        logging.info(
            'Working in %s environment', self.Parameters.environment)
        logging.info(
            'Using DynamicPoiCalculatorExecutable %s',
            self.calculator_binary.id)

    def _choose_ecstatic_environments(self):
        if self.Parameters.environment == 'testing':
            self.ecstatic_environments = ['datatesting']
        elif self.Parameters.environment == 'stable':
            self.ecstatic_environments = ['stable', 'testing']
        else:
            raise ValueError(
                'Unknown environment: {}'.format(self.Parameters.environment))

    def on_execute(self):
        self._ensure_latest_resources_used()
        self._choose_ecstatic_environments()
        self._load_executable(self.calculator_binary.id, 'calculator')

        current_datetime = datetime.strftime(self.created, DATETIME_FORMAT)
        if self.Parameters.forced_datetime:
            current_datetime = str(self.Parameters.forced_datetime)

        env = {
            'YT_TOKEN': self.Parameters.yt_token.data(),
            'YT_PROXY': 'hahn',
            'JUGGLER_ENABLE': str(int(self.juggler_report_is_enabled)),
            'JUGGLER_HOST': self.juggler_host_name,
        }
        if self.Parameters.juggler_token:
            env['JUGGLER_TOKEN'] = self.Parameters.juggler_token.data()

        os.environ.update(env)

        subprocess.check_call('mkdir data', shell=True)
        with self.memoize_stage.calculator(commit_on_entrance=False):
            sources = dict(self.Parameters.dynamic_poi_sources)
            cmd = [
                './calculator',
                '--sources',
            ]
            cmd.extend(sources.values())
            cmd.append('--sources-names')
            cmd.extend(sources.keys())
            cmd.extend([
                '--resources-history',
                self.Parameters.resources_history_yt_file,
            ])
            cmd.extend([
                '--output-fb-file',
                'data/dynamic_poi.fb',
                '--output-table',
                os.path.join(self.Parameters.output_yt_table_prefix, 'latest')
            ])
            logging.info('Running command %s', ' '.join(cmd))
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('calculator')) as pl:
                subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)

        version = '{}-{}'.format(
            current_datetime.replace('-', '.'),
            str(self.Parameters.version)
        )
        with open('data/version', 'w') as f:
            f.write(version)


        try:
            for env in self.ecstatic_environments:
                args = ['upload', '{}={}'.format(DATASET_NAME, version),
                        'data', '+{}'.format(BRANCH)]
                tvm_id = TVM_ID_DATATESTING if env == 'datatesting' else TVM_ID_STABLE
                tvm_secret_id = TVM_SECRET_ID_DATATESTING if env == 'datatesting' else TVM_SECRET_ID_STABLE

                self.ecstatic(
                    env,
                    args,
                    tvm_id=tvm_id,
                    tvm_secret_id=tvm_secret_id,
                )
        except subprocess.CalledProcessError as e:
            self.set_info(e.output)
            raise TaskFailure('Ecstatic returned ' + str(e.returncode))
