from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.maps.common.ecstatic_bin import MapsEcstaticToolMixin

import subprocess
import shutil

from datetime import datetime
from os import environ, mkdir


DATASET_NAME = 'yandex-maps-zero-speed-banner-data'
ECSTATIC_DATE_FORMAT = '%y.%m.%d'
TVM_ID_TESTING = 2019591
TVM_SECRET_ID_TESTING = 'sec-01e59w5yt777r035bz79t2h2pt'
TVM_ID_STABLE = 2019593
TVM_SECRET_ID_STABLE = 'sec-01e59w6b0pwf68geg20hdbw9vx'


class EdgeStatsPackerExecutable(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class ZeroSpeedBannerModel(sdk2.Resource):
    releasable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class MapsUploadZeroSpeedBannerData(MapsEcstaticToolMixin, sdk2.Task):
    """Pack edge_stats table into fbs. Upload stats and model to ecstatic"""
    class Requirements(sdk2.Task.Requirements):
        ram = 5120  # 5 GB
        disk_space = 5120 # 5 GB

    class Parameters(sdk2.Parameters):
        owner = 'MAPS'
        # NOTE: ecstatic branch should be the same as env
        # (testing -> testing; stable -> stable)
        environments = sdk2.parameters.List('Ecstatic environments', default=['stable', 'testing'])
        version = sdk2.parameters.String(
            'version',
            default='{}-0'.format(datetime.strftime(datetime.today(), ECSTATIC_DATE_FORMAT))
        )
        yt_token = sdk2.parameters.Vault('yt token vault', required=True)

        edge_stats_table = sdk2.parameters.String(
            'Path to edge_stats table',
            required=True
        )
        packer_binary = sdk2.parameters.Resource(
            'Sandbox resource ID for packer binary',
            resource_type=EdgeStatsPackerExecutable,
            required=True
        )
        model_file = sdk2.parameters.Resource(
            'Sandbox resource ID for catboost model',
            resource_type=ZeroSpeedBannerModel,
            required=True
        )

    def on_execute(self):
        binary = sdk2.ResourceData(sdk2.Resource[self.Parameters.packer_binary])
        model = sdk2.ResourceData(sdk2.Resource[self.Parameters.model_file])

        shutil.copyfile(str(binary.path), './binary')
        environ['YT_TOKEN'] = self.Parameters.yt_token.data()
        cmd = [
            './binary',
            '--edge-stats-table',
            self.Parameters.edge_stats_table
        ]
        subprocess.check_call('chmod +x ./binary', shell=True)
        subprocess.check_call(cmd)
        mkdir('data')
        shutil.move('edge_stats.fb', 'data')
        shutil.copyfile(str(model.path), './data/model.cbm')
        try:
            for env in self.Parameters.environments:
                args = ['upload', '{}={}'.format(DATASET_NAME, str(self.Parameters.version)),
                        'data', '+stable']
                tvm_id = TVM_ID_STABLE if env == 'stable' else TVM_ID_TESTING
                tvm_secret_id = TVM_SECRET_ID_STABLE if env == 'stable' else TVM_SECRET_ID_TESTING
                self.ecstatic(
                    env,
                    args,
                    tvm_id=tvm_id,
                    tvm_secret_id=tvm_secret_id
                )
        except subprocess.CalledProcessError as e:
            self.set_info(e.output)
            raise TaskFailure('Ecstatic returned ' + str(e.returncode))
