from datetime import datetime, timedelta
import logging
import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.maps.common.latest_resources import find_latest_resource
from sandbox.projects.maps.common.juggler_alerts import TaskJugglerReportWithParameters
from sandbox.projects.maps.MapsCalculateZeroSpeedBanerStats import (
    CLUSTER,
    TTIMES_DATE_FORMAT,
    MapsCalculateZeroSpeedBanerStats
)
from sandbox.projects.maps.MapsUploadZeroSpeedBannerData import (
    ECSTATIC_DATE_FORMAT,
    EdgeStatsPackerExecutable,
    MapsUploadZeroSpeedBannerData,
    ZeroSpeedBannerModel
)
from sandbox.sandboxsdk import environments


class MapsZeroSpeedBannerPipeline(TaskJugglerReportWithParameters):
    """Run Zero Speed Banner pipeline"""
    class Requirements(sdk2.Task.Requirements):
        environments = [environments.PipEnvironment('yandex-yt')]

    class Parameters(TaskJugglerReportWithParameters.Parameters):
        with sdk2.parameters.RadioGroup('Environment') as environment:
            environment.values["testing"] = environment.Value("testing", default=True)
            environment.values["stable"] = environment.Value("stable")

        owner = 'MAPS'

        with sdk2.parameters.Group('ecstatic parameters') as ecstatic_parameters:
            environments = sdk2.parameters.List('Ecstatic environments', default=['stable', 'testing'])

        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            yt_token = sdk2.parameters.Vault('yt token vault', required=True)
            yql_token = sdk2.parameters.Vault('yql token vault', required=True)

        with sdk2.parameters.Group('Common parameters') as common_parameters:
            num_days = sdk2.parameters.Integer(
                'num days to process before (including) the end date',
                default=14,
                required=True
            )
            packer_binary = sdk2.parameters.Resource(
                'Sandbox resource ID for packer binary',
                resource_type=EdgeStatsPackerExecutable,
                required=False
            )
            model_file = sdk2.parameters.Resource(
                'Sandbox resource ID for catboost model',
                resource_type=ZeroSpeedBannerModel,
                required=True
            )

    def ensure_latest_resources_used(self):
        self.packer_binary = self.Parameters.packer_binary
        if not self.packer_binary:
            self.packer_binary = find_latest_resource(
                EdgeStatsPackerExecutable, self.Parameters.environment)

        logging.info(
            'Working in %s environment', self.Parameters.environment)
        logging.info(
            'Using EdgeStatsPackerExecutable: %s',
            self.packer_binary.id)

    def on_execute(self):
        import yt.wrapper as yt
        yt.config['proxy']['url'] = CLUSTER
        yt.config['token'] = self.Parameters.yt_token.data()

        ecstatic_version = '{}-0'.format(datetime.strftime(datetime.today(), ECSTATIC_DATE_FORMAT))
        # Note: usually travel time have delay in 3 days
        ttimes_end_date = datetime.strftime(datetime.today() - timedelta(days=4), TTIMES_DATE_FORMAT)

        self.ensure_latest_resources_used()

        with self.memoize_stage.calc_edge_stats:
            self.Context.edge_stats_table = yt.create_temp_table()
            calc_edge_stats_task = MapsCalculateZeroSpeedBanerStats(
                self,
                create_sub_task=False,
                yt_token=self.Parameters.yt_token,
                yql_token=self.Parameters.yql_token,
                end_date=ttimes_end_date,
                num_days=self.Parameters.num_days,
                output_table=self.Context.edge_stats_table
            ).enqueue()
            raise sdk2.WaitTask(
                [calc_edge_stats_task.id],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True,
                timeout=14400  # 4 hours
            )

        with self.memoize_stage.upload_data:
            upload_data_task = MapsUploadZeroSpeedBannerData(
                self,
                create_sub_task=False,
                environments=self.Parameters.environments,
                version=ecstatic_version,
                yt_token=self.Parameters.yt_token,
                edge_stats_table=self.Context.edge_stats_table,
                packer_binary=self.packer_binary,
                model_file=self.Parameters.model_file
            ).enqueue()
            raise sdk2.WaitTask(
                [upload_data_task.id],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True,
                timeout=14400  # 4 hours
            )

        for subtask in list(self.find(children=True)):
            if subtask.status != ctt.Status.SUCCESS:
                raise TaskError('subtask failed')
