import logging
import os
import shutil
import stat
import subprocess

from sandbox import sdk2
from sandbox.projects.maps.common.juggler_alerts import (
    TaskJugglerReportWithParameters
)


TVM_SECRET_YAV_ID_TESTING = 'sec-01f3t9k03mq2srsv6wnwg5az17'
TVM_SECRET_YAV_ID_STABLE = 'sec-01f3t9knga18249dwxvaex8epq'


class NewsfeedHypothesesGenerationExecutable(sdk2.Resource):
    executable = True
    releasable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class AddressExtractorData(sdk2.Resource):
    """
        Facts data for address extractor
        rus_geo_addr.proto rus_geo_addr_in_web.rules.bin rus_geo_dict.gzt.bin
        https://a.yandex-team.ru/arc/trunk/arcadia/FactExtract/address_extractor
    """
    auto_backup = True
    releasable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class NewsfeedHypothesesGeneration(TaskJugglerReportWithParameters):
    # requrements for MULTISLOT tag
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 4 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(TaskJugglerReportWithParameters.Parameters):
        kill_timeout = 24 * 60 * 60  # 24 hours

        with sdk2.parameters.Group('Executable resource') as executable_resource:
            executable = sdk2.parameters.Resource(
                'Sandbox resource ID of the executable',
                resource_type=NewsfeedHypothesesGenerationExecutable,
                required=True
            )

        with sdk2.parameters.Group('Executable parameters') as input_parameters:
            with sdk2.parameters.RadioGroup('Environment to post hypotheses') as environment:
                environment.values.testing = environment.Value(value='testing', default=True)
                environment.values.stable = environment.Value(value='stable')

            probability = sdk2.parameters.Float(
                'Probability threshold', default=0.5
            )
            min_publish_timestamp = sdk2.parameters.Integer(
                'Minimum publication timestamp', default=0
            )
            processing_interval = sdk2.parameters.Integer(
                'News processing interval (minutes)', default=30
            )
            hours_to_run = sdk2.parameters.Integer(
                'Total time to run (hours)', default=23
            )
            min_news_freshness = sdk2.parameters.Integer(
                'News freshness (hours)', default=1
            )
            keywords_IoU_threshold = sdk2.parameters.Float(
                'IoU threshold for keywords deduplication', default=0.5
            )
            address_extractor_data = sdk2.parameters.Resource(
                'Address extractor data',
                resource_type=AddressExtractorData,
                required=True
            )

        with sdk2.parameters.Group('YT tokens') as yt_tokens:
            yt_vault = sdk2.parameters.Vault('YT vault', required=True)

    def on_execute(self):
        resource = sdk2.Resource[self.Parameters.executable]
        resource_data = sdk2.ResourceData(resource)
        executable_path = './executable'
        shutil.copyfile(str(resource_data.path), executable_path)
        os.chmod(
            executable_path, os.stat(executable_path).st_mode | stat.S_IEXEC
        )

        os.environ['YT_TOKEN'] = self.Parameters.yt_vault.data()
        tvm_secret_id = (TVM_SECRET_YAV_ID_STABLE
                         if self.Parameters.environment == 'stable'
                         else TVM_SECRET_YAV_ID_TESTING)
        os.environ['TVM_SECRET'] = sdk2.yav.Secret(tvm_secret_id).data()['client_secret']

        address_extractor_data = sdk2.ResourceData(self.Parameters.address_extractor_data)
        address_extractor_data_folder = sdk2.path.Path('rus_geo_addr')
        address_extractor_data_folder.mkdir()
        untar_address_extractor_data_cmd = [
            'tar', '--extract',
            '--file', str(address_extractor_data.path),
            '--directory', str(address_extractor_data_folder),
            '--verbose'
        ]
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('untar_address_extractor_data')) as pl:
            subprocess.check_call(untar_address_extractor_data_cmd, stdout=pl.stdout, stderr=subprocess.STDOUT)

        cmd = [
            executable_path,
            '--probability',
            str(self.Parameters.probability),
            '--min-publish-timestamp',
            str(self.Parameters.min_publish_timestamp),
            '--processing-interval',
            str(self.Parameters.processing_interval),
            '--hours-to-run',
            str(self.Parameters.hours_to_run),
            '--min-news-freshness',
            str(self.Parameters.min_news_freshness),
            '--keywords-iou-threshold',
            str(self.Parameters.keywords_IoU_threshold),
            '--environment',
            str(self.Parameters.environment),
        ]

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('executable')) as pl:
            subprocess.check_call(cmd, stdout=pl.stdout, stderr=subprocess.STDOUT)
