import cStringIO
import csv
import logging
import os
import subprocess

from sandbox import sdk2
from sandbox.sandboxsdk import environments
import sandbox.common.types.resource as ctr


class OsmRoadHypothesesGeneratorExecutable(sdk2.Resource):
    executable = True
    releasable = True
    releasers = ['tswr']


class OsmRoadHypothesesTsvOutput(sdk2.Resource):
    pass


def to_tsv(columns, data, defaults):
    ss = cStringIO.StringIO()
    writer = csv.writer(ss, delimiter='\t')
    writer.writerow(columns)
    unpack = lambda row: [row.get(c, defaults.get(c)) for c in columns]
    writer.writerows(map(unpack, data))
    return ss.getvalue()


def table_to_tsv_resource(ytc, table, columns, defaults, resource):
    data = ytc.read_table(table)
    resource.write_bytes(to_tsv(columns, data, defaults))


class OsmRoadHypothesesGenerationTask(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        ]

    class Parameters(sdk2.Task.Parameters):
        description_text = sdk2.parameters.String(
            'Text description of hypotheses', default='Maybe some roads are missing here',
            required=True)
        with sdk2.parameters.Group('Bounding box') as bbox_block:
            north = sdk2.parameters.String('North', required=True)
            south = sdk2.parameters.String('South', required=True)
            east = sdk2.parameters.String('East', required=True)
            west = sdk2.parameters.String('West', required=True)
        with sdk2.parameters.Group('Hypotheses params') as hyp_params_block:
            ignore_road_types = sdk2.parameters.String(
                    'Road types to ignore', default='unclassified,tertiary')
            threshold = sdk2.parameters.String(
                    'Threshold (meters) for lumping', default='15', required=True)
        with sdk2.parameters.Group('YT settings') as yt_block:
            yt_prefix = sdk2.parameters.String(
                    'yt-prefix', required=True)
            name = sdk2.parameters.String('Yt table name', required=True)
            yt_vault_token = sdk2.parameters.String(
                    'Your token name in vault', required=True)
        with sdk2.parameters.Group('Resources') as resources_block:
            resource_id = sdk2.parameters.LastReleasedResource(
                    'YT executable',
                    resource_type=OsmRoadHypothesesGeneratorExecutable,
                    state=(ctr.State.READY,),
                    required=True)
            container = sdk2.parameters.Container(
                    'LXC container', required=True)

    def _upload_to_yt(self):
        logging.info('Start osm export and upload to yt')
        src_dir = os.path.dirname(os.path.realpath(__file__))
        script_path = os.path.join(src_dir, 'upload_roads_from_osm_to_yt.py')

        with sdk2.helpers.ProcessLog(self, logger='upload-log') as pl:
            env = os.environ.copy()
            yt_token = sdk2.Vault.data(self.owner,
                                       self.Parameters.yt_vault_token)
            env['YT_TOKEN'] = yt_token
            ignore = ''
            if self.Parameters.ignore_road_types:
                ignore = '--ignore-road-types={}'.format(
                        self.Parameters.ignore_road_types)
            subprocess.check_call(
                ['/py3/bin/python', script_path,
                    '--output-table', self.roads_table,
                    '--north', self.Parameters.north,
                    '--south', self.Parameters.south,
                    '--east', self.Parameters.east,
                    '--west', self.Parameters.west,
                    ignore],
                stdout=pl.stdout, stderr=pl.stdout, env=env)

    def _generate_hypotheses(self):
        logging.info('Start generating hypotheses')
        runner = self.Parameters.resource_id
        if runner is None:
            raise errors.TaskError('No executable founded')
        runner = sdk2.ResourceData(runner)

        with sdk2.helpers.ProcessLog(self, logger='hypotheses-log') as pl:
            env = os.environ.copy()
            env['YT_TOKEN'] = sdk2.Vault.data(self.owner,
                                              self.Parameters.yt_vault_token)
            subprocess.check_call(
                [str(runner.path),
                    '--input-table', self.roads_table,
                    '--output-table', self.hypotheses_table,
                    '--threshold', self.Parameters.threshold],
                stdout=pl.stdout, stderr=pl.stdout, env=env)

    def _dump_hypotheses_to_resource(self):
        logging.info('Start osm export and upload to yt')
        import yt.wrapper as yt
        yt_proxy = 'hahn'
        yt_token = sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token)
        ytc = yt.YtClient(proxy=yt_proxy, token=yt_token)
        output = OsmRoadHypothesesTsvOutput(
            self, 'Output file',
            '{}.txt'.format(os.path.basename(self.hypotheses_table)))
        resource = sdk2.ResourceData(output)
        defaults = {'description': self.Parameters.description_text}
        table_to_tsv_resource(
                ytc, self.hypotheses_table, ['lon', 'lat', 'description'],
                defaults, resource.path)
        resource.ready()

    def on_execute(self):
        self.roads_table = os.path.join(self.Parameters.yt_prefix,
                                        self.Parameters.name)
        self.hypotheses_table = self.roads_table + '_hypotheses'

        self._upload_to_yt()
        self._generate_hypotheses()
        self._dump_hypotheses_to_resource()
